// Impulse Response Processor model implementation
//
// Teru KAMOGASHIRA

#include "irmodel.hpp"

irmodel::irmodel()
{
  current = 0;
  currentProcessSampleSize = 0;
  scalewet = 3.0f;
  scaledry = 2.0f;
  wet = 1/scalewet;
  dry = 0.0f;
  width = 1.0f;
  update();
  sampleSize = 0;
}

irmodel::~irmodel()
{
  freeFFT();
  freeImpulse();
}

void irmodel::mute(float * f, int t)
{
  for(int i = 0;i < t;i ++)
    f[i] = 0.0f;
}

void irmodel::inverseImpulse()
{
  ;
}

void irmodel::loadImpulse(float * inputL, float * inputR, int numsamples)
{
  int inverse = 0;
  impulseSize = numsamples;
  int pulse = checkPow2(numsamples);
#ifdef DEBUG
  fprintf(stderr, "%d->%d\n", numsamples, pulse);
#endif
  allocImpulse(pulse);
  
  if(inverse == 0)
    {
      memcpy(impulseL, inputL, sizeof(float)*numsamples);
      memcpy(impulseR, inputR, sizeof(float)*numsamples);
    }
  else
    {
      for(int i = 0;i < numsamples;i ++)
	{
	  impulseL[i] = inputL[numsamples-i];
	  impulseR[i] = inputR[numsamples-i];
	}
    }
  mute(delaylineL, impulseSize*2);
  mute(delaylineR, impulseSize*2);
  mute(fifoL, sampleSize*3);
  mute(fifoR, sampleSize*3);
  current = 0;

  fftwf_plan planL, planR;
  planL = fftwf_plan_r2r_1d(2*pulse, impulseL, fftImpulseL, FFTW_R2HC, FFTW_ESTIMATE);
  planR = fftwf_plan_r2r_1d(2*pulse, impulseR, fftImpulseR, FFTW_R2HC, FFTW_ESTIMATE);
  fftwf_execute(planL);
  fftwf_execute(planR);
  fftwf_destroy_plan(planL);
  fftwf_destroy_plan(planR);

  if(currentProcessSampleSize != pulse)
    allocFFT(pulse);

  delayL.setsize(numsamples);
  delayL.mute();
  delayR.setsize(numsamples);
  delayR.mute();
}

int irmodel::getSampleSize()
{
  return sampleSize;
}

int irmodel::getLatency()
{
  return delayL.getsize();
}

void irmodel::processreplace(float *inputL, float *inputR, float *outputL, float *outputR,
                              long numsamples, int skip)
{
  process(inputL, inputR, outputL, outputR, numsamples, skip, 'R');
}

void irmodel::processmix(float *inputL, float *inputR, float *outputL, float *outputR,
                          long numsamples, int skip)
{
  process(inputL, inputR, outputL, outputR, numsamples, skip, 'M');
}

void irmodel::process(float *inputL, float *inputR, float *outputL, float *outputR,
		      long numsamples, int skip, char pmode)
{
  int pro = 0;
  if(numsamples == 0)
    return;
  if(numsamples <= impulseSize)
    {
      memcpy(&(fifoL[fifoSize+impulseSize]), inputL, sizeof(float)*numsamples);
      memcpy(&(fifoR[fifoSize+impulseSize]), inputR, sizeof(float)*numsamples);
      if(fifoSize+numsamples >= impulseSize)
	{
	  processSquare(&(fifoL[impulseSize]), &(fifoR[impulseSize]),
			&(fifoL[impulseSize]), &(fifoR[impulseSize]));
	}
      for(int i = 0;i < numsamples;i ++)
	{
	  switch(pmode)
	    {
	    case 'M':
	      // mixing
	      outputL[i] = fifoL[i+fifoSize]*wet1 + fifoR[i+fifoSize]*wet2 + delayL.process(inputL[i])*dry;
	      outputR[i] = fifoR[i+fifoSize]*wet1 + fifoL[i+fifoSize]*wet2 + delayR.process(inputR[i])*dry;
          break;
	    case 'R':
	    default:
	      // replacing
	      outputL[i] = fifoL[i+fifoSize]*wet1 + fifoR[i+fifoSize]*wet2 + delayL.process(inputL[i])*dry;
	      outputR[i] = fifoR[i+fifoSize]*wet1 + fifoL[i+fifoSize]*wet2 + delayR.process(inputR[i])*dry;
	      break;
	    }
	}
      fifoSize += numsamples;
      if(fifoSize >= impulseSize)
	{
	  memmove(fifoL, &(fifoL[impulseSize]), sizeof(float)*2*impulseSize);
	  memmove(fifoR, &(fifoR[impulseSize]), sizeof(float)*2*impulseSize);
	  fifoSize -= impulseSize;
	}
    }
  else
    {
      // divide
      int div = numsamples/(impulseSize);
      for(int i = 0;i < div;i ++)
	{
	  process(&(inputL[i*impulseSize]), &(inputR[i*impulseSize]),
		  &(outputL[i*impulseSize]), &(outputR[i*impulseSize]),
		  impulseSize, skip, pmode);
	}
      process(&(inputL[div*impulseSize]), &(inputR[div*impulseSize]),
	      &(outputL[div*impulseSize]), &(outputR[div*impulseSize]),
	      numsamples%(impulseSize), skip, pmode);
    }
}

void irmodel::processSquare(float *inputL, float *inputR, float *outputL, float *outputR)
{
  mute(fftOrigL, sampleSize*2);
  mute(fftOrigR, sampleSize*2);
  memcpy(fftOrigL, inputL, sizeof(float)*impulseSize);
  memcpy(fftOrigR, inputR, sizeof(float)*impulseSize);
  
  fftwf_execute(planOrigL);
  fftwf_execute(planOrigR);
  
  // mul
  fftRevL[0] = fftOrigL[0] * fftImpulseL[0];
  fftRevR[0] = fftOrigR[0] * fftImpulseR[0];
  fftRevL[sampleSize] = fftOrigL[sampleSize] * fftImpulseL[sampleSize];
  fftRevR[sampleSize] = fftOrigR[sampleSize] * fftImpulseR[sampleSize];
  for(int i = 1;i < sampleSize;i ++)
    {
      {
	float e = fftOrigL[i];
	float d = fftOrigL[sampleSize*2 - i];
	float f = fftImpulseL[i];
	float g = fftImpulseL[sampleSize*2 - i];
	fftRevL[2*sampleSize-i] = e*g + f*d;
	fftRevL[i] = e*f - d*g;
      }
      {
	float e = fftOrigR[i];
	float d = fftOrigR[sampleSize*2 - i];
	float f = fftImpulseR[i];
	float g = fftImpulseR[sampleSize*2 - i];
	fftRevR[2*sampleSize-i] = e*g + f*d;
	fftRevR[i] = e*f - d*g;
      }
    }
  
  fftwf_execute(planRevL);
  fftwf_execute(planRevR);
  for(int i = 0;i < impulseSize*2;i ++)
    {
      fftRevL[i] /= impulseSize*0x10;
      fftRevR[i] /= impulseSize*0x10;
    }
  
  // sigma
  // XXXXOOOO
  // OXXXXOOO
  // OOXXXXOO
  // OOOXXXXO
  // 11112222
  // -------
  // --- ----
  // ^
  //     ^
  if(current == 0)
    {
      for(int i = 0;i < impulseSize*2-1;i ++)
	{
	  delaylineL[i] += fftRevL[i];
	  delaylineR[i] += fftRevR[i];
	}

      memcpy(outputL, delaylineL, sizeof(float)*impulseSize);
      memcpy(outputR, delaylineR, sizeof(float)*impulseSize);
      mute(delaylineL, impulseSize);
      mute(delaylineR, impulseSize);
      current = impulseSize;
    }
  else // current == impulseSize
    {
      for(int i = 0;i < impulseSize;i ++)
	{
	  delaylineL[impulseSize+i] += fftRevL[i];
	  delaylineR[impulseSize+i] += fftRevR[i];
	}
      for(int i = 0;i < impulseSize-1;i ++)
	{
	  delaylineL[i] += fftRevL[impulseSize+i];
	  delaylineR[i] += fftRevR[impulseSize+i];
	}
      memcpy(outputL, &(delaylineL[impulseSize]), sizeof(float)*impulseSize);
      memcpy(outputR, &(delaylineR[impulseSize]), sizeof(float)*impulseSize);
      mute(&(delaylineL[impulseSize]), impulseSize);
      mute(&(delaylineR[impulseSize]), impulseSize);
      current = 0;
    }
}

void irmodel::allocImpulse(int numsamples)
{
  freeImpulse();
  impulseL = new float[2*numsamples];
  impulseR = new float[2*numsamples];
  fifoL = new float[3*numsamples];
  fifoR = new float[3*numsamples];
  fifoSize = numsamples;
  delaylineL = new float[2*impulseSize];
  delaylineR = new float[2*impulseSize];
  sampleSize = numsamples;
  fftImpulseL = (float*)fftwf_malloc(sizeof(float)*2*numsamples);
  fftImpulseR = (float*)fftwf_malloc(sizeof(float)*2*numsamples);
}

void irmodel::freeImpulse()
{
  if(sampleSize != 0)
    {
      delete[] fifoL;
      delete[] fifoR;
      delete[] delaylineL;
      delete[] delaylineR;
      delete[] impulseL;
      delete[] impulseR;
      fftwf_free(fftImpulseL);
      fftwf_free(fftImpulseR);
      sampleSize = 0;
    }
}

void irmodel::allocFFT(int numsamples)
{
  freeFFT();
  fftRevL = (float*)fftwf_malloc(sizeof(float)*2*numsamples);
  fftRevR = (float*)fftwf_malloc(sizeof(float)*2*numsamples);
  fftOrigL = (float*)fftwf_malloc(sizeof(float)*2*numsamples);
  fftOrigR = (float*)fftwf_malloc(sizeof(float)*2*numsamples);
  planOrigL = fftwf_plan_r2r_1d(2*numsamples, fftOrigL, fftOrigL, FFTW_R2HC, FFTW_MEASURE);
  planOrigR = fftwf_plan_r2r_1d(2*numsamples, fftOrigR, fftOrigR, FFTW_R2HC, FFTW_MEASURE);
  planRevL = fftwf_plan_r2r_1d(2*numsamples, fftRevL, fftRevL, FFTW_HC2R, FFTW_MEASURE);
  planRevR = fftwf_plan_r2r_1d(2*numsamples, fftRevR, fftRevR, FFTW_HC2R, FFTW_MEASURE);
  currentProcessSampleSize = numsamples;
  mute(fftRevL, sampleSize*2);
  mute(fftRevR, sampleSize*2);
  mute(fftOrigL, sampleSize*2);
  mute(fftOrigR, sampleSize*2);
}

void irmodel::freeFFT()
{
  if(currentProcessSampleSize != 0)
    {
      fftwf_free(fftRevL);
      fftwf_free(fftRevR);
      fftwf_destroy_plan(planRevL);
      fftwf_destroy_plan(planRevR);
      fftwf_free(fftOrigL);
      fftwf_free(fftOrigR);
      fftwf_destroy_plan(planOrigL);
      fftwf_destroy_plan(planOrigR);
    }
}

void irmodel::update()
{
  wet1 = wet*(width/2 + 0.5f);
  wet2 = wet*((1-width)/2);
}

void irmodel::setwet(float value)
{
  wet = value*scalewet;
  update();
}

float irmodel::getwet()
{
  return wet/scalewet;
}

void irmodel::setdry(float value)
{
  dry = value*scaledry;
}

float irmodel::getdry()
{
  return dry/scaledry;
}

void irmodel::setwidth(float value)
{
  width = value;
  update();
}

float irmodel::getwidth()
{
  return width;
}
