// NReverb model
// Copyright (C) 2006 Teru KAMOGASHIRA

#include "nrev.hpp"

nrev::nrev()
{
  defaultfs = 25641;
  feedback = 0.7f;
  damp2 = 0.7f;
  damp3 = 0.3f;
  numcombs = 0;
  numallpasses = 0;
  currentfs = defaultfs;
  setFilter(1);
  setDefault();
  // Buffer will be full of rubbish - so we MUST mute them
  mute();
}

nrev::~nrev()
{
  freeFilter();
}

void nrev::resetfs(int fs)
{
  currentfs = fs;
  float tf = (float)fs/(float)defaultfs;
#ifdef DEBUG
  fprintf(stderr, "freeverb: Fs=%d[Hz] Factor=%f\n", fs, tf);
#endif
  setFilter(tf);
  setDefault();
  mute();
}

int nrev::f_(int def, float factor)
{
  return (int)((float)(def)*factor);
}

int nrev::p_(int def, float factor)
{
  int base = (int)((float)(def)*factor);
  while(!isPrime(base)) base++;
  return base;
}

bool nrev::isPrime(int number)
{
  if (number == 2) return true;
  if (number & 1)       {
    for (int i=3; i<(int)sqrt((double)number)+1; i+=2)
      if ( (number % i) == 0) return false;
    return true; // prime
  }
  else return false; // even
}

void nrev::printconfig()
{
  fprintf(stderr, "*** Freeverb config ***\n");
  fprintf(stderr, "Fs=%d[Hz]\n",currentfs);
  fprintf(stderr, "gain %f roomsize %f roomsize1 %f\n",
	  gain, roomsize,roomsize1);
  fprintf(stderr, "damp %f damp1 %f damp2 %f damp3 %f wet %f wet1 %f wet2 %f\n",
	  damp,damp1,damp2,damp3,wet,wet1,wet2);
  fprintf(stderr, "dry %f width %f mode %f\n", dry,width,mode);
}

void nrev::mute()
{
  if (getmode() >= freezemode)
    return;
  
  for (int i=0;i<numcombs;i++)
    {
      combL[i].mute();
      combR[i].mute();
    }
  for (int i=0;i<numallpasses;i++)
    {
      allpassL[i].mute();
      allpassR[i].mute();
    }
}

void nrev::processreplace(float *inputL, float *inputR, float *outputL, float *outputR,
			      long numsamples, int skip)
{
  process(inputL, inputR, outputL, outputR, numsamples, skip, 'R');
}

void nrev::processmix(float *inputL, float *inputR, float *outputL, float *outputR,
			  long numsamples, int skip)
{
  process(inputL, inputR, outputL, outputR, numsamples, skip, 'M');
}

void nrev::process(float *inputL, float *inputR, float *outputL, float *outputR,
		     long numsamples, int skip, char pmode)
{
  float outC,outL,outR,input;
  
  while(numsamples-- > 0)
    {
      input = (*inputL + *inputR) * gain;
      outL = outR = outC = 0.0;

      // highpass filter
      static float hpf = 0.0;
      float hpfCof = 0.3;
      hpfCof = damp3;
      input = hpf = -hpfCof*hpf + (1-hpfCof)*input;
      
      // Accumulate comb filters in parallel
      for(int i=0; i<numcombs; i++)
	{
	  outL += combL[i].process(input);
	  outR += combR[i].process(input);
	}
      
      // Feed through allpasses in series
      for(int i=0; i<3; i++)
	{
	  outL = allpassL[i].process(outL);
	  outR = allpassR[i].process(outR);
	}
      
      // lowpass filter
      static float lpfL = 0.0, lpfR = 0.0;
      float lpfCof = 0.7;
      lpfCof = damp2;
      lpfL = lpfCof*lpfL + (1-lpfCof)*outL;
      lpfR = lpfCof*lpfR + (1-lpfCof)*outR;
      outL = lpfL;
      outR = lpfR;
      
      outL = allpassL[3].process(outL);
      outR = allpassR[3].process(outR);
      //outRearL = allpassL[4].process(outL);
      //outRearR = allpassR[4].process(outR);
      
      outL = allpassL[5].process(outL);
      outR = allpassL[6].process(outR);
      //outRearL = allpassL[7].process(outRearL);
      //outRearR = allpassL[8].process(outRearR);

      switch(pmode)
	{
	case 'M':
	  // mixing
	  *outputL += outL*wet1 + outR*wet2 + *inputL*dry;
	  *outputR += outR*wet1 + outL*wet2 + *inputR*dry;
	  break;
	case 'R':
	  // replacing
	  *outputL = outL*wet1 + outR*wet2 + *inputL*dry;
	  *outputR = outR*wet1 + outL*wet2 + *inputR*dry;
	  break;
	default:
	  break;
	}
      
      // Increment sample pointers, allowing for interleave (if any)
      inputL += skip;
      inputR += skip;
      outputL += skip;
      outputR += skip;
    }
}

void nrev::update()
{
  // Recalculate internal values after parameter change
  
  int i;
  
  wet1 = wet*(width/2 + 0.5f);
  wet2 = wet*((1-width)/2);
  
  if (mode >= freezemode)
    {
      roomsize1 = 1;
      damp1 = 0;
      gain = muted;
    }
  else
    {
      roomsize1 = roomsize;
      damp1 = damp;
      gain = fixedgain;
    }
  
  for(i=0; i<numcombs; i++)
    {
      float fbl = pow(10.0, (-3 * (float)combL[i].getsize() / (roomsize1 * currentfs)));
      combL[i].setfeedback(fbl);
      float fbr = pow(10.0, (-3 * (float)combR[i].getsize() / (roomsize1 * currentfs)));
      combR[i].setfeedback(fbr);
    }
  
  for(i=0; i<numcombs; i++)
    {
      combL[i].setdamp(damp1);
      combR[i].setdamp(damp1);
    }
}

void nrev::setroomsize(float value)
{
  roomsize = value;
  update();
}

float nrev::getroomsize()
{
  return roomsize;
}
void nrev::setfeedback(float value)
{
  feedback = value;
  for(int i=0; i<numallpasses; i++)
    {
      allpassL[i].setfeedback(value);
      allpassR[i].setfeedback(value);
    }
}

float nrev::getfeedback()
{
  return feedback;
}

void nrev::setdamp(float value)
{
  damp = value;
  update();
}

float nrev::getdamp()
{
  return damp;
}

void nrev::setdamp2(float value)
{
  damp2 = value;
}

float nrev::getdamp2()
{
  return damp2;
}

void nrev::setdamp3(float value)
{
  damp3 = value;
}

float nrev::getdamp3()
{
  return damp3;
}

void nrev::setwet(float value)
{
  wet = value*scalewet;
  update();
}

float nrev::getwet()
{
  return wet/scalewet;
}

void nrev::setdry(float value)
{
  dry = value*scaledry;
}

float nrev::getdry()
{
  return dry/scaledry;
}

void nrev::setwidth(float value)
{
  width = value;
  update();
}

float nrev::getwidth()
{
  return width;
}

void nrev::setmode(float value)
{
  mode = value;
  update();
}

float nrev::getmode()
{
  if (mode >= freezemode)
    return 1;
  else
    return 0;
}

void nrev::allocFilter(int ncomb, int nallpass)
{
  if(numcombs != 0||numallpasses != 0)
    freeFilter();
  combL = new comb[ncomb];
  combR = new comb[ncomb];
  allpassL = new allpass[nallpass];
  allpassR = new allpass[nallpass];
  numcombs = ncomb;
  numallpasses = nallpass;
}

void nrev::freeFilter()
{
  if(numcombs != 0&&numallpasses != 0)
    {
#ifdef DEBUG
  fprintf(stderr, "nrev::freeFilter()+\n");
#endif
      delete[] combL;
      delete[] combR;
      delete[] allpassL;
      delete[] allpassR;
      numcombs = 0;
      numallpasses = 0;
    }
}

void nrev::setFilter(float factor)
{
#ifdef DEBUG
  fprintf(stderr, "nrev::setBuffer(%f)\n", factor);
#endif
  const int combT[] = {1433, 1601, 1867, 2053, 2251, 2399};
  const int allpT[] = {347, 113, 37, 59, 53, 43, 37, 29, 19};
  //                  (0    1    2) F3  R4  L5  R6 RL7  RR8
  
  allocFilter(sizeof(combT)/sizeof(int), sizeof(allpT)/sizeof(int));
  for(int i=0; i<numcombs; i++)
    {
      combL[i].setsize(p_(combT[i],factor));
      combR[i].setsize(p_(f_(combT[i],factor)+f_(stereospread441*defaultfs/44100,factor),1));
    }
  for(int i=0; i<numallpasses; i++)
    {
      allpassL[i].setsize(p_(allpT[i],factor));
      allpassR[i].setsize(p_(f_(allpT[i],factor)+f_(stereospread441*defaultfs/44100,factor),1));
    }
}

void nrev::setDefault()
{
  for(int i=0; i<numallpasses; i++)
    {
      allpassL[i].setfeedback(0.7f);
      allpassR[i].setfeedback(0.7f);
    }
  setwet(initialwet);
  setroomsize(initialroom);
  setdry(initialdry);
  setdamp(initialdamp);
  setdamp2(0.7f);
  setwidth(initialwidth);
  setmode(initialmode);
}

//ends
