#include "libSGP.h"
#include "midi.h"
#include "md.h"
#include "glib.h"
#include "elements.h"
#include <map>

#define BUFSIZE 0x10000

libSGP sgp;
struct rootElement * root;
struct sequenceState * seq;
struct element * el;
unsigned long end_time;

unsigned char GMSystemOn[] =  {0xf0, 0x7e, 0x7f, 0x09, 0x01, 0xf7};
unsigned char GMSystemOff[] = {0xf0, 0x7e, 0x7f, 0x09, 0x02, 0xf7};
unsigned char GM2SystemOn[] = {0xf0, 0x7e, 0x7f, 0x09, 0x03, 0xf7};
unsigned char GSReset[] =     {0xF0, 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7F, 0x00, 0x41, 0xF7};
//                                            ^device number
unsigned char XGSystemOn[] =  {0xF0, 0x43, 0x10, 0x4C, 0x00, 0x00, 0x7E, 0x00, 0xF7};
//                                            ^device number
unsigned char MUBasic[] =     {0xF0, 0x43, 0x10, 0x49, 0x00, 0x00, 0x12, 0x00, 0xF7};
//                                            ^device number
unsigned char MU100Native[] = {0xF0, 0x43, 0x10, 0x49, 0x00, 0x00, 0x12, 0x01, 0xF7};
//                                            ^device number

// must not be zero
double time_base = 480;
double micro_tempo = 100000;

// 44100/1000000 (unkown)
double bytePar = 0.044100;
// 1~50 (unkown)
guint32 interval = 16;

typedef struct{
  guint32 element_time;
  short device_channel;
  short note;  
} NoteOff;

std::multimap<guint32,struct element*> elements;
std::multimap<guint32,NoteOff> noteOff;
std::multimap<guint32,double> tempoChange;
std::vector<MFrame> mFrames;

long int timeToByte(guint32 element_time)
{
  double el_time = (double)element_time;
  double par = bytePar*el_time*micro_tempo/time_base;
  return (long int)par;
}

void insertEventSysEx(std::vector<MFrame> * mFrame, long int relativeByte,
		      unsigned char id, unsigned char * data, int length)
{
  MFrame mf;
  mf.relativeByte = relativeByte;
  //mf.midiMessage.push_back(0xF0);
  mf.midiMessage.push_back(id);
  for(int i = 0;i < length;i ++)
    {
      mf.midiMessage.push_back(data[i]);
    }
  mFrame->push_back(mf);
}

void insertEvent(std::vector<MFrame> * mFrame, long int relativeByte,
		 unsigned char c1, unsigned char c2, unsigned char c3)
{
  //fprintf(stderr, "{%d}", relativeByte);
  //fprintf(stderr, "%02x %02x %02x\n", c1, c2, c3);
  MFrame mf;
  mf.relativeByte = relativeByte;
  mf.midiMessage.push_back(c1);
  mf.midiMessage.push_back(c2);
  mf.midiMessage.push_back(c3);
  mFrame->push_back(mf);
}

void play(struct element *el, std::vector<MFrame> * mFrame, guint32 frameByte)
{
  //seq_midi_event_init(ctxp, &ev, el->element_time, el->device_channel);
  unsigned char LSB, MSB;
  switch (el->type) {
  case MD_TYPE_ROOT:
    fprintf(stderr, "\n[time_base = %d]\n", MD_ROOT(el)->time_base);
    //time_base = (double)MD_ROOT(el)->time_base;
    //seq_init_tempo(ctxp, MD_ROOT(el)->time_base, 120, 1);
    //seq_start_timer(ctxp);
    break;

  case MD_TYPE_NOTE: // 9nH ΡȥʥС ٥ƥ (note on)
    insertEvent(mFrame, frameByte,
		0x90+(el->device_channel), MD_NOTE(el)->note, MD_NOTE(el)->vel);
    //seq_midi_note(ctxp, &ev, el->device_channel, MD_NOTE(el)->note, MD_NOTE(el)->vel, MD_NOTE(el)->length);
    break;

  case MD_TYPE_CONTROL: // BnH ȥʥС ǡ
    insertEvent(mFrame, frameByte,
		0xB0+(el->device_channel), MD_CONTROL(el)->control, MD_CONTROL(el)->value);
    //seq_midi_control(ctxp, &ev, el->device_channel, MD_CONTROL(el)->control, MD_CONTROL(el)->value);
    break;
  
  case MD_TYPE_PROGRAM: // CnH ץʥС
    insertEvent(mFrame, frameByte,
		0xC0+(el->device_channel), MD_PROGRAM(el)->program, 0x0);
    //seq_midi_program(ctxp, &ev, el->device_channel, MD_PROGRAM(el)->program);
    break;

  case MD_TYPE_TEMPO: //
    fprintf(stderr, "\n[micro_tempo = %d]\n", MD_TEMPO(el)->micro_tempo);
    //next_micro_tempo = (double);
    //seq_midi_tempo(ctxp, &ev, MD_TEMPO(el)->micro_tempo);
    break;
    
  case MD_TYPE_PITCH: // EnH ǡLSB˥ǡMSB
    // 00H 00HΤȤäȤԥå
    // ʥԥå٥ɥۥּ˰ư֡ԥå٥ɥСֺޤݤ֡
    // 00H 40HǴԥå
    // ʥԥå٥ɥۥ롿СȤˤ֡
    // 7FH 7FHǤäȤԥå夬
    // ʥԥå٥ɥۥָޤǰư֡ԥå٥ɥСֱޤݤ֡ˤޤ

    /* MD_PITCH(el)->pitch was centered around zero. */
    short int pitch_ = MD_PITCH(el)->pitch + 0x2000;
    if(pitch_ < 0x80)
      {
	LSB = pitch_;
	MSB = 0;
      }
    else
      {
	LSB = pitch_%0x80;
	MSB = (pitch_-LSB)/0x80;
      }
    insertEvent(mFrame, frameByte,
		0xE0+(el->device_channel), LSB, MSB);
    //seq_midi_pitchbend(ctxp, &ev, el->device_channel, MD_PITCH(el)->pitch);
    break;
    
  case MD_TYPE_PRESSURE: // DnH ץå㡼
    insertEvent(mFrame, frameByte,
		0xD0+(el->device_channel), MD_PRESSURE(el)->velocity, 0x0);
    //seq_midi_chanpress(ctxp, &ev, el->device_channel, MD_PRESSURE(el)->velocity);
    break;

  case MD_TYPE_KEYTOUCH: // AnH ΡȥʥС ץå㡼
    insertEvent(mFrame, frameByte,
		0xA0+(el->device_channel), MD_KEYTOUCH(el)->note, MD_KEYTOUCH(el)->velocity);
    //seq_midi_keypress(ctxp, &ev, el->device_channel, MD_KEYTOUCH(el)->note, MD_KEYTOUCH(el)->velocity);
    break;

  case MD_TYPE_SYSEX: // F0H ᡼ID ǡǤĹ
    insertEventSysEx(mFrame, frameByte,
		     MD_SYSEX(el)->status, MD_SYSEX(el)->data, MD_SYSEX(el)->length);
    //seq_midi_sysex(ctxp, &ev, MD_SYSEX(el)->status, MD_SYSEX(el)->data,
    //MD_SYSEX(el)->length);
    break;

  case MD_TYPE_TEXT:
  case MD_TYPE_KEYSIG:
  case MD_TYPE_TIMESIG:
  case MD_TYPE_SMPTEOFFSET:
    /* Ones that have no sequencer action */
    break;
  default:
    printf("WARNING: play: not implemented yet %d\n", el->type);
    break;
  }
}

int main(int argc, char * argv[])
{
  char * FileName = NULL;
  char * ProcNameOrOrdinal = NULL;
  char * MidiFileName = NULL;
  //char dFileName[] = "SGP.DLL";
  //char dProcNameOrOrdinal[] = "main";

  if(argc > 3)
    {
      fprintf(stderr, "Using %s(%s)\nfile:%s\n", argv[1], argv[2], argv[3]);
      FileName = argv[1];
      ProcNameOrOrdinal = argv[2];
      MidiFileName = argv[3];
    }
  else
    {
      fprintf(stderr, "Usage: %s FileName ProcNameOrOrdinal MidiFile\n", argv[0]);
      return 0;
    }
  
  if(sgp.loadSgpDll(FileName, ProcNameOrOrdinal) != 0)
    return -1;
  
  if (strcmp(MidiFileName, "-") == 0)
    root = midi_read(stdin);
  else
    root = midi_read_file(MidiFileName);
  if (!root)
    return -1;
  
  seq = md_sequence_init(root);
  
  // build multimap and register note offs
  while((el = md_sequence_next(seq)) != NULL)
    {
      if(el->type == MD_TYPE_NOTE)
	{
	  NoteOff nf;
	  nf.element_time = el->element_time + MD_NOTE(el)->length;
	  nf.device_channel = el->device_channel;
	  nf.note = MD_NOTE(el)->note;
	  noteOff.insert(std::pair<guint32,NoteOff>(el->element_time+MD_NOTE(el)->length,nf));
	}
      else if(el->type == MD_TYPE_TEMPO)
	{
	  tempoChange.insert(std::pair<guint32,double>(el->element_time,
						       (double)MD_TEMPO(el)->micro_tempo));
	}
      else if(el->type == MD_TYPE_ROOT)
	{
	  time_base = (double)MD_ROOT(el)->time_base;
	}
      elements.insert(std::pair<guint32,struct element *>(el->element_time,el));
    }
  // Get the end time for the tracks and echo an event to
  // wake us up at that time
  end_time = md_sequence_end_time(seq);
  fprintf(stderr, "[end_time = %d]\n", end_time);

  // Loop through all the elements in the song and play them
  guint32 readByte = 0;

  unsigned char * L = new unsigned char[BUFSIZE*4];
  unsigned char * R = new unsigned char[BUFSIZE*4];
  
  sgp.eventCall(0x23, 0, 0,      0, 0);
  sgp.eventCall(0x00, 0, 0,      0, 0);
  sgp.eventCall(0x02, 0, 0,      0, 0);
  sgp.eventCall(0x0A, 0, 0,      0, 0x47C24400);
  sgp.eventCall(0x0B, 0, BUFSIZE,0, 0); // Default ReadData Range
  sgp.eventCall(0x0C, 0, 0x1,    0, 0);
  sgp.eventCall(0x02, 0, 0,      0, 0);

  for(guint32 i = interval;i < (end_time + interval);i += interval)
    {
      // 1. events except note off
      std::multimap<guint32,struct element *>::iterator eLBegin = elements.lower_bound(0);
      std::multimap<guint32,struct element *>::iterator eLEnd = elements.upper_bound(i);
      for(std::multimap<guint32,struct element *>::iterator eIter = eLBegin;eIter != eLEnd;++eIter)
	{
	  if(timeToByte(eIter->first)-readByte < 0)
	    exit(-1);
	  play(eIter->second, &mFrames,
	       timeToByte(eIter->first)-readByte);
	  elements.erase(eIter);
	}
      
      // 2. note offs
      std::multimap<guint32,NoteOff>::iterator cIBegin = noteOff.lower_bound(0);
      std::multimap<guint32,NoteOff>::iterator cIEnd = noteOff.upper_bound(i);
      for(std::multimap<guint32,NoteOff>::iterator cIter = cIBegin;cIter != cIEnd;++cIter)
	{
	  if(timeToByte(cIter->first)-readByte < 0)
	    exit(-1);
	  // 8nH ΡȥʥС ٥ƥ (note off)
	  insertEvent(&mFrames,
		      timeToByte(cIter->first)-readByte,
		      0x80+(cIter->second.device_channel), cIter->second.note, 0x0);
	  noteOff.erase(cIter);
	}
      
      fprintf(stderr, "{0x%08x=%10d|0x%08x[0x%04x]%3d(0x%08x,0x%08x)}\r",
	      timeToByte(i), i, readByte, timeToByte(i)-readByte, mFrames.size(),
	      elements.size(), noteOff.size());
      sgp.playFrame(mFrames, L, R, timeToByte(i)-readByte);
      sgp.dumpLR(L, R, timeToByte(i)-readByte);
      mFrames.clear();
      readByte += timeToByte(i)-readByte;
      
      // 0. tempo change
      std::multimap<guint32,double>::iterator tLBegin = tempoChange.lower_bound(0);
      std::multimap<guint32,double>::iterator tLEnd = tempoChange.upper_bound(i);
      for(std::multimap<guint32,double>::iterator tIter = tLBegin;tIter != tLEnd;++tIter)
	{
	  micro_tempo = tIter->second;
	  readByte = timeToByte(i);
	  tempoChange.erase(tIter);
	}
    }
  
  //seq_midi_echo(end);
  md_free(MD_ELEMENT(root));
  
  fprintf(stderr, "\n");
  fprintf(stderr, "left %d element(s).\n", elements.size());
  fprintf(stderr, "left %d note off(s).\n", noteOff.size());
  
  sgp.eventCall(0x0C, 0, 0x0, 0, 0);
  sgp.eventCall(0x01, 0, 0x0, 0, 0);
  //sgp.unloadSgpDll();
  return 0;
}
