#include "libSGP.h"
#include "midi.h"
#include "md.h"
#include "glib.h"
#include "elements.h"
#include <map>
#include <nrev.hpp>
#include <math.h>

#define BUFSIZE 0x100000
#define RESTTIME 100

//#define MIN(x,y) (x<y?x:y)
//#define MAX(x,y) (x>y?x:y)

float BaseSamplingRate = 96000.0;

libSGP sgp;
int voice = 0, maxvoice = 0;
struct rootElement * root;
struct sequenceState * seq;
struct element * el;
unsigned long end_time;

unsigned char GMSystemOn[]  = {0x7e, 0x7f, 0x09, 0x01, 0xf7};
unsigned char GMSystemOff[] = {0x7e, 0x7f, 0x09, 0x02, 0xf7};
unsigned char GM2SystemOn[] = {0x7e, 0x7f, 0x09, 0x03, 0xf7};
unsigned char GSReset[]     = {0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7F, 0x00, 0x41, 0xF7};
//                                      ^device number
unsigned char XGSystemOn[]  = {0x43, 0x10, 0x4C, 0x00, 0x00, 0x7E, 0x00, 0xF7};
//                                      ^device number
unsigned char MUBasic[]     = {0x43, 0x10, 0x49, 0x00, 0x00, 0x12, 0x00, 0xF7};
//                                      ^device number
unsigned char MU100Native[] = {0x43, 0x10, 0x49, 0x00, 0x00, 0x12, 0x01, 0xF7};
//                                      ^device number

// must not be zero
double time_base = 480;
double micro_tempo = 100000;

double bytePar = BaseSamplingRate/1000000;
// 1~50 (unkown)
guint32 interval = 16;

typedef struct{
  guint32 element_time;
  short device_channel;
  short note;  
} NoteOff;

std::multimap<guint32,struct element*> elements;
std::multimap<guint32,NoteOff> noteOff;
std::multimap<guint32,double> tempoChange;
std::vector<MFrame> mFrames;

int ecmp(const unsigned char * ref, unsigned char * sysex,
	      int reflen, int datalen)
{
  return memcmp(ref,sysex,MIN(reflen,datalen));
}

void checkReset(unsigned char * sysEx, int len)
{
  if(ecmp(GMSystemOn,sysEx,sizeof(GMSystemOn),len) == 0)
    {
      fprintf(stderr, "\n<<GMSystemOn>>\n");
    }
  if(ecmp(GM2SystemOn,sysEx,sizeof(GM2SystemOn),len) == 0)
    {
      fprintf(stderr, "\n<<GM2SystemOn>>\n");
    }
  if(ecmp(GSReset,sysEx,sizeof(GSReset),len) == 0)
    {
      fprintf(stderr, "\n<<GSReset>>\n");
    }
  if(ecmp(XGSystemOn,sysEx,sizeof(XGSystemOn),len) == 0)
    {
      fprintf(stderr, "\n<<XGSystemOn>>\n");
    }
  if(ecmp(MUBasic,sysEx,sizeof(MUBasic),len) == 0)
    {
      fprintf(stderr, "\n<<MUBasic>>\n");
    }
  if(ecmp(MU100Native,sysEx,sizeof(MU100Native),len) == 0)
    {
      fprintf(stderr, "\n<<MU100Native>>\n");
    }
}

// meter
float meter(unsigned char * L, unsigned char * R, int len)
{
  float maxL=0.0, maxR=0.0;
  int c = 0;
  float *_L = (float*)L;
  float *_R = (float*)R;
  
  for(c = 0;c < len;c ++)if(fabsf(_L[c]) > maxL) maxL = _L[c];
  for(c = 0;c < len;c ++)if(fabsf(_R[c]) > maxL) maxL = _R[c];
  return((maxL+maxR)/2.0);
}

#define BARLENGTH 30
void printbar(float meter)
{
  int bar = (int)(meter*BARLENGTH);
  int space = BARLENGTH-bar;
  for(int c = 0;c < bar;c ++)
    fprintf(stderr, "#");
  for(int c = 0;c < space;c ++)
    fprintf(stderr, " ");
}

// reverb
nrev reverbm;

// (*Mix_EffectFunc_t)
void reverb(unsigned char * L, unsigned char * R, int len)
{
  int c = 0;
  float * f;
  
  f = (float*)L;
  for(c = 0;c < len;c ++)f[c] *= 0.5;
  f = (float*)R;
  for(c = 0;c < len;c ++)f[c] *= 0.5;
  //reverbm.processreplace(fL,fR,oL,oR,len,1);
  reverbm.processmix((float*)L,(float*)R,(float*)L,(float*)R,len,1);
}

long int timeToByte(guint32 element_time)
{
  double el_time = (double)element_time;
  double par = bytePar*el_time*micro_tempo/time_base;
  return (long int)par;
}

void insertEventSysEx(std::vector<MFrame> * mFrame, long int relativeByte,
		      unsigned char id, unsigned char * data, int length)
{
  checkReset(data, length);
  MFrame mf;
  mf.relativeByte = relativeByte;
  //mf.midiMessage.push_back(0xF0);
  mf.midiMessage.push_back(id);
  for(int i = 0;i < length;i ++)
    {
      mf.midiMessage.push_back(data[i]);
    }
  mFrame->push_back(mf);
}

void insertEvent(std::vector<MFrame> * mFrame, long int relativeByte,
		 unsigned char c1, unsigned char c2, unsigned char c3)
{
  //fprintf(stderr, "{%d}", relativeByte);
  //fprintf(stderr, "%02x %02x %02x\n", c1, c2, c3);
  MFrame mf;
  mf.relativeByte = relativeByte;
  mf.midiMessage.push_back(c1);
  mf.midiMessage.push_back(c2);
  mf.midiMessage.push_back(c3);
  mFrame->push_back(mf);
}

void play(struct element *el, std::vector<MFrame> * mFrame, guint32 frameByte)
{
  //seq_midi_event_init(ctxp, &ev, el->element_time, el->device_channel);
  unsigned char LSB, MSB;
  switch (el->type) {
  case MD_TYPE_ROOT:
    fprintf(stderr, "[time_base = %d]", MD_ROOT(el)->time_base);
    //time_base = (double)MD_ROOT(el)->time_base;
    //seq_init_tempo(ctxp, MD_ROOT(el)->time_base, 120, 1);
    //seq_start_timer(ctxp);
    break;

  case MD_TYPE_NOTE: // 9nH ΡȥʥС ٥ƥ (note on)
    insertEvent(mFrame, frameByte,
		0x90+(el->device_channel), MD_NOTE(el)->note, MD_NOTE(el)->vel);
    voice ++;
    if(maxvoice < voice)
      maxvoice = voice;
    //seq_midi_note(ctxp, &ev, el->device_channel, MD_NOTE(el)->note, MD_NOTE(el)->vel, MD_NOTE(el)->length);
    break;

  case MD_TYPE_CONTROL: // BnH ȥʥС ǡ
    insertEvent(mFrame, frameByte,
		0xB0+(el->device_channel), MD_CONTROL(el)->control, MD_CONTROL(el)->value);
    //seq_midi_control(ctxp, &ev, el->device_channel, MD_CONTROL(el)->control, MD_CONTROL(el)->value);
    break;
  
  case MD_TYPE_PROGRAM: // CnH ץʥС
    insertEvent(mFrame, frameByte,
		0xC0+(el->device_channel), MD_PROGRAM(el)->program, 0x0);
    //seq_midi_program(ctxp, &ev, el->device_channel, MD_PROGRAM(el)->program);
    break;

  case MD_TYPE_TEMPO: //
    fprintf(stderr, "\n[micro_tempo = %d]\n", MD_TEMPO(el)->micro_tempo);
    //next_micro_tempo = (double);
    //seq_midi_tempo(ctxp, &ev, MD_TEMPO(el)->micro_tempo);
    break;
    
  case MD_TYPE_PITCH: // EnH ǡLSB˥ǡMSB
    // 00H 00HΤȤäȤԥå
    // ʥԥå٥ɥۥּ˰ư֡ԥå٥ɥСֺޤݤ֡
    // 00H 40HǴԥå
    // ʥԥå٥ɥۥ롿СȤˤ֡
    // 7FH 7FHǤäȤԥå夬
    // ʥԥå٥ɥۥָޤǰư֡ԥå٥ɥСֱޤݤ֡ˤޤ

    /* MD_PITCH(el)->pitch was centered around zero. */
    short int pitch_ = MD_PITCH(el)->pitch + 0x2000;
    if(pitch_ < 0x80)
      {
	LSB = pitch_;
	MSB = 0;
      }
    else
      {
	LSB = pitch_%0x80;
	MSB = (pitch_-LSB)/0x80;
      }
    insertEvent(mFrame, frameByte,
		0xE0+(el->device_channel), LSB, MSB);
    //seq_midi_pitchbend(ctxp, &ev, el->device_channel, MD_PITCH(el)->pitch);
    break;
    
  case MD_TYPE_PRESSURE: // DnH ץå㡼
    insertEvent(mFrame, frameByte,
		0xD0+(el->device_channel), MD_PRESSURE(el)->velocity, 0x0);
    //seq_midi_chanpress(ctxp, &ev, el->device_channel, MD_PRESSURE(el)->velocity);
    break;

  case MD_TYPE_KEYTOUCH: // AnH ΡȥʥС ץå㡼
    insertEvent(mFrame, frameByte,
		0xA0+(el->device_channel), MD_KEYTOUCH(el)->note, MD_KEYTOUCH(el)->velocity);
    //seq_midi_keypress(ctxp, &ev, el->device_channel, MD_KEYTOUCH(el)->note, MD_KEYTOUCH(el)->velocity);
    break;

  case MD_TYPE_SYSEX: // F0H ᡼ID ǡǤĹ
    insertEventSysEx(mFrame, frameByte,
		     MD_SYSEX(el)->status, MD_SYSEX(el)->data, MD_SYSEX(el)->length);
    //seq_midi_sysex(ctxp, &ev, MD_SYSEX(el)->status, MD_SYSEX(el)->data,
    //MD_SYSEX(el)->length);
    break;

  case MD_TYPE_TEXT:
  case MD_TYPE_KEYSIG:
  case MD_TYPE_TIMESIG:
  case MD_TYPE_SMPTEOFFSET:
    /* Ones that have no sequencer action */
    break;
  default:
    fprintf(stderr, "WARNING: play: not implemented yet %d\n", el->type);
    break;
  }
}

int main(int argc, char * argv[])
{
  char * FileName = NULL;
  char * ProcNameOrOrdinal = NULL;
  char * MidiFileName = NULL;
  //char dFileName[] = "SGP.DLL";
  //char dProcNameOrOrdinal[] = "main";
  FILE * fp = stdout;
  
  if(argc > 4)
    {
      fprintf(stderr, "Using %s(%s)\nfile:%s\n", argv[1], argv[2], argv[3]);
      FileName = argv[1];
      ProcNameOrOrdinal = argv[2];
      MidiFileName = argv[3];
      if(strcmp(argv[4], "-") != 0)
	{
	  fp = fopen(argv[4], "wb");
	  if(fp == NULL)
	    {
	      fprintf(stderr, "!fopen(%s,wb)\n", argv[4]);
	      return -1;
	    }
	}
    }
  else
    {
      fprintf(stderr, "Usage: %s dll main MidiFile OutFile(stdout if -)\n", argv[0]);
      return 0;
    }
  /*
    if((sgp.checkSYXG50(1) + sgp.checkSYXG50L(1) + sgp.checkSYXG50C(1)) != 0)
    {
    fprintf(stderr, "!checkSYXG50 failed.\n");
    }
  */
    
  if(sgp.loadSgpDll(FileName, ProcNameOrOrdinal) != 0)
    return -1;

 loop:
  
  if (strcmp(MidiFileName, "-") == 0)
    root = midi_read(stdin);
  else
    root = midi_read_file(MidiFileName);
  if (!root)
    return -1;
  
  seq = md_sequence_init(root);
  
  // build multimap and register note offs
  while((el = md_sequence_next(seq)) != NULL)
    {
      if(el->type == MD_TYPE_NOTE)
	{
	  NoteOff nf;
	  nf.element_time = el->element_time + MD_NOTE(el)->length;
	  nf.device_channel = el->device_channel;
	  nf.note = MD_NOTE(el)->note;
	  noteOff.insert(std::pair<guint32,NoteOff>(el->element_time+MD_NOTE(el)->length,nf));
	}
      else if(el->type == MD_TYPE_TEMPO)
	{
	  tempoChange.insert(std::pair<guint32,double>(el->element_time,
						       (double)MD_TEMPO(el)->micro_tempo));
	}
      else if(el->type == MD_TYPE_ROOT)
	{
	  time_base = (double)MD_ROOT(el)->time_base;
	}
      elements.insert(std::pair<guint32,struct element *>(el->element_time,el));
    }
  // Get the end time for the tracks and echo an event to
  // wake us up at that time
  end_time = md_sequence_end_time(seq);
#ifdef DEBUG
  fprintf(stderr, "[end_time = %08x]\n", end_time);
#endif
  // Loop through all the elements in the song and play them
  guint32 readByte = 0;

  unsigned char * L = new unsigned char[BUFSIZE*4];
  unsigned char * R = new unsigned char[BUFSIZE*4];
  
  long int SampleRateL = 0x472C4400; // when SR = 44100
  memcpy(&SampleRateL, &BaseSamplingRate, sizeof(BaseSamplingRate));
  fprintf(stderr, "sampling rate = %f(0x%08x)\n",
	  BaseSamplingRate, SampleRateL);

  // reverb
  reverbm.resetfs((int)BaseSamplingRate);
  reverbm.setwet(.5);
  reverbm.setroomsize(2.05);
  reverbm.setfeedback(0.65);
  reverbm.setdry(0.1);
  reverbm.setdamp(0.2);
  reverbm.setdamp2(0.7);
  reverbm.setdamp3(0.15);
  reverbm.setwidth(0.7);
  reverbm.printconfig();
  
  // eventCall start
  
  sgp.eventCall(0x00, 0, 0,      0, 0);
  sgp.eventCall(0x02, 0, 0,      0, 0);
  sgp.eventCall(0x0A, 0, 0,      0, SampleRateL);
  sgp.eventCall(0x0B, 0, BUFSIZE,0, 0); // Default ReadData Range
  sgp.eventCall(0x0C, 0, 0x1,    0, 0);
  sgp.eventCall(0x02, 0, 0,      0, 0);

  for(guint32 i = interval;i < (end_time + RESTTIME*interval);i += interval)
    {
      // 1. events except note off
      std::multimap<guint32,struct element *>::iterator eLBegin = elements.lower_bound(0);
      std::multimap<guint32,struct element *>::iterator eLEnd = elements.upper_bound(i);
      for(std::multimap<guint32,struct element *>::iterator eIter = eLBegin;eIter != eLEnd;++eIter)
	{
	  if(timeToByte(eIter->first)-readByte < 0)
	    exit(-1);
	  play(eIter->second, &mFrames,
	       timeToByte(eIter->first)-readByte);
	  elements.erase(eIter);
	}
      
      // 2. note offs
      std::multimap<guint32,NoteOff>::iterator cIBegin = noteOff.lower_bound(0);
      std::multimap<guint32,NoteOff>::iterator cIEnd = noteOff.upper_bound(i);
      for(std::multimap<guint32,NoteOff>::iterator cIter = cIBegin;cIter != cIEnd;++cIter)
	{
	  if(timeToByte(cIter->first)-readByte < 0)
	    exit(-1);
	  // 8nH ΡȥʥС ٥ƥ (note off)
	  insertEvent(&mFrames,
		      timeToByte(cIter->first)-readByte,
		      0x80+(cIter->second.device_channel), cIter->second.note, 0x0);
	  noteOff.erase(cIter);
	  voice --;
	}
      
#ifdef DEBUG
      fprintf(stderr,
	      "{0x%08x=%10d|0x%08x[0x%04x]%3d(0x%08x,0x%08x)}[%3d|%3d]\r",
	      timeToByte(i), i, readByte, timeToByte(i)-readByte, mFrames.size(),
	      elements.size(), noteOff.size(),voice,maxvoice);
#else
      fprintf(stderr, "[%3d|%3d]%10d/%10d\r", voice, maxvoice, i, end_time);
#endif
      sgp.playFrame(mFrames, L, R, timeToByte(i)-readByte);
      // reverb
      reverb(L, R, timeToByte(i)-readByte);
#ifndef DEBUG
      //float meter_t = meter(L, R, timeToByte(i)-readByte);
      //printbar(meter_t);
      //fprintf(stderr, "%f\r", meter_t);
#endif
      sgp.dumpLR_S32(L, R, timeToByte(i)-readByte, fp);
      mFrames.clear();
      readByte += timeToByte(i)-readByte;
      
      // 0. tempo change
      std::multimap<guint32,double>::iterator tLBegin = tempoChange.lower_bound(0);
      std::multimap<guint32,double>::iterator tLEnd = tempoChange.upper_bound(i);
      for(std::multimap<guint32,double>::iterator tIter = tLBegin;tIter != tLEnd;++tIter)
	{
	  micro_tempo = tIter->second;
	  readByte = timeToByte(i);
	  tempoChange.erase(tIter);
	}
    }
  
  //seq_midi_echo(end);
  md_free(MD_ELEMENT(root));
  
  //goto loop;

  fprintf(stderr, "\n");
  fprintf(stderr, "left %d element(s).\n", elements.size());
  fprintf(stderr, "left %d note off(s).\n", noteOff.size());
  
  sgp.eventCall(0x0C, 0, 0x0, 0, 0);
  sgp.eventCall(0x01, 0, 0x0, 0, 0);
  //sgp.unloadSgpDll();
  return 0;
}
