/*
 * 
 *  libSGP.cpp - VSTi loading code
 * 
 *  Copyright (C) 2006 Teru KAMOGASHIRA
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "libSGP.h"

int __cdecl libSGP::sgpInit(SGPInit sgp)
{
  char yid[] = "YAMAHA";
  char cid[] = "Cubase VST";
#ifdef DEBUG
  fprintf(stderr, "sgpInit():<%08x>[%08x]<%08x><%08x><%08x><%08x>\n",
	  sgp.r1, sgp.mode, sgp.r3, sgp.r4, sgp.r5, sgp.r6);
#endif
  switch(sgp.mode)
    {
    case 0x1:
      return 2;
      
    case 0x2:
      return 0x6D723220;
      
    case 0x3:
      return 1;
      
    case 0x20:
      memcpy(sgp.r5, yid, sizeof(yid));
      return 0;

    case 0x21:
      memcpy(sgp.r5, cid, sizeof(cid));
      return 0;

    case 0x22:
      return 0x3E8;

    case 0xE:
      
    case 0x2A:

    case 0x26:
      return 1;
      
    default:
      return 0;
    }
}

int libSGP::loadSgpDll(char * dllPath, char * procNameOrOrdinal)
{
  typedef int __cdecl (*SGPI)(SGPInit);
  typedef SGPFunctionTable*(*SGPMAIN)(SGPI);
  SGPMAIN sgpMain;
  
  if((SGPInstance = LoadLibraryA(dllPath)) == NULL)
    {
      fprintf(stderr, "!LoadLibraryA(%s)\n", dllPath);
      return -1;
    }
  if((sgpMain = (SGPMAIN)GetProcAddress(SGPInstance, procNameOrOrdinal)) == NULL)
    {
      fprintf(stderr, "!GetProcAddress(%s)\n", procNameOrOrdinal);
      return -1;
    }
  sgpFunctionTable = sgpMain(&libSGP::sgpInit);
  if(sgpFunctionTable == 0)
    {
      fprintf(stderr, "!%s Internal Error or Version Mismatch.\n", dllPath);
      return -1;
    }
  
  unsigned char vendorString[64] = "";
  unsigned char productString[64] = "";
  eventCall(0x2f, 0, 0, (int)vendorString, 0);
  fprintf(stderr, "VendorString = <%s> ", vendorString);
  eventCall(0x30, 0, 0, (int)productString, 0);
  fprintf(stderr, "ProductString = <%s>\n", productString);
  
#ifdef DEBUG
  fprintf(stderr, "sgpFunctionTable at 0x%08x\n", sgpFunctionTable);
  for(long int i_ = 0;i_ < sgpFunctionTable->r6;i_ ++)
    {
      char progName[24];
      eventCall(5,0,0,(int)progName,0);
      fprintf(stderr, "ProgramName %d %s\n", i_, progName);
    }
  for(long int i_ = 0;i_ < sgpFunctionTable->r7;i_ ++)
    {
      typedef float (*TA)(SGPFunctionTable *,long int);
      TA ta = (TA)sgpFunctionTable->r5;
      float f = ta(sgpFunctionTable, i_);
      fprintf(stderr, "getParameter %d %f\n", i_, f);
    }
  fprintf(stderr, "numParams\t0x%08x\n", sgpFunctionTable->r7);
  fprintf(stderr, "effectflags\t0x%08x\n", sgpFunctionTable->r12);
  fprintf(stderr, "par automation\t%d\n", eventCall(26,0,0,0,0));
  fprintf(stderr, "vstversion\t%d\n", eventCall(58,0,0,0,0));
  fprintf(stderr, "plugcategory\t0x%08x\n", eventCall(0x23, 0, 0, 0, 0));
#endif
  
  unsigned char * chunk;
  long int chunkSize = eventCall(23,0,0,(int)&chunk,0);
#ifdef DEBUG
  fprintf(stderr, "chunkSize\t0x%08x\n", chunkSize);
  for(int i_ = 0;i_ < chunkSize;i_ ++)
    {
      fprintf(stderr, "0x%02x,", (unsigned char)chunk[i_]);
      if(i_%0x10 == 0xf)
	fprintf(stderr, "\n");
    }
  fprintf(stderr, "\n");
#endif
  // S-YG20.DLL
  // 00 6b 74 65 72 6d 00 4a 4c 45 53 53 43 48 41 52 
  // 53 45 54 3d 65 75 63 2d 6a 70 00 57 40 01 01 01 
  // 65 
  // S-YXG50.DLL
  // 44_65_66_61_75_6c_74_20_53_65_74_75_70_00_41 52 
  // 53 45 54 3d 65 75 63 2d 6a 70 00 57 20_01 01 01 
  // 65
  unsigned char syxgChunkHeadPoly[] =
    {0x44,0x65,0x66,0x61,0x75,0x6c,0x74,0x20,
     0x53,0x65,0x74,0x75,0x70,0x00,0x41,0x52,
     0x53,0x45,0x54,0x3d,0x65,0x75,0x63,0x2d,
     0x6a,0x70,0x00,0x57,0x40,0x01,0x01,0x01,
     0x65,};
  unsigned char syxgProductString[64] = "YAMAHA XG SoftSynthesizer S-YXG50";
  unsigned char hqProductString[64] = "HQ Software Synthesizer";
  if(strcmp((const char*)productString, (const char*)syxgProductString) == 0)
    {
      fprintf(stderr, "S-YXG50 detected.\n");
      chunk[0x1C] = 0x80; // polyphony 128=0x80
      chunk[0x20] = 0x65; // volume max=7F normal=65
      eventCall(24,0,chunkSize,(int)chunk,0);
    }
  else if(chunkSize == 0x21)
    {
      fprintf(stderr, "Might be S-YG20.\n");
      syxgChunkHeadPoly[0x1C] = 0x80; // polyphony 128=0x80
      syxgChunkHeadPoly[0x20] = 0x65; // volume max=7F normal=65
      eventCall(24,0,chunkSize,(int)syxgChunkHeadPoly,0);
    }
  else if(strcmp((const char*)productString, (const char*)hqProductString) == 0)
    {
      fprintf(stderr, "Hyper Canvas detected.\n");
    }
  return 0;
}

int libSGP::MidiReadData(ReadData * readData)
{
  typedef int (*TA)(ReadData);
  TA ta = (int (*)(ReadData))sgpFunctionTable->r8;
  return ta(*readData);
}

int libSGP::MidiAddData(ReadData * readData)
{
  typedef int (*TA)(ReadData);
  TA ta = (int (*)(ReadData))sgpFunctionTable->r3;
  return ta(*readData);
}

int libSGP::MidiEventCall(EventCall * eventCall)
{
  typedef int (*TA)(EventCall);
  TA ta = (int (*)(EventCall))sgpFunctionTable->r2;
  return ta(*eventCall);
}

int libSGP::eventCall(int eventCode, int r2, int r3, int r4, int r5)
{
  EventCall eventCall;
  eventCall.sgpFunctionTable = sgpFunctionTable;
  eventCall.r2 = (void *)eventCode;
  eventCall.r3 = (void *)r2;
  eventCall.r4 = (void *)r3;
  eventCall.r5 = (MInfo *)r4;
  eventCall.r6 = (void *)r5;
  return MidiEventCall(&eventCall);
}

int libSGP::playFrame(std::vector<MFrame> mFrame,
		      unsigned char * L, unsigned char * R, int size)
{
  // build MInfo
  void ** mInfo = new void*[mFrame.size()+2];
  mInfo[0] = (void *)mFrame.size();
  mInfo[1] = (void *)0xBAADF00D;
  MidiMessage * midiMessages = new MidiMessage[mFrame.size()];

  for(int i = 0;i < (int)mFrame.size();i ++)
    {
      mInfo[2+i] = (void *)&(midiMessages[i]);
      midiMessages[i].byte = mFrame[i].relativeByte;
      midiMessages[i].r2 = 0x18;
      midiMessages[i].r4 = 0x0;
      midiMessages[i].r6 = 0x0;
      if(mFrame[i].midiMessage.size() <= 0x3)
	{
	  midiMessages[i].mode = 0x1;
	  midiMessages[i].SysExLength = 0x0;
	  memcpy(&(midiMessages[i].dataOrSysEx),
		 mFrame[i].midiMessage.c_str(),
		 mFrame[i].midiMessage.size());
	}
      else
	{
	  midiMessages[i].mode = 0x6;
	  midiMessages[i].SysExLength = mFrame[i].midiMessage.size();
	  midiMessages[i].dataOrSysEx = (void *)mFrame[i].midiMessage.c_str();
	}
    }
  
  EventCall eventCall;
  eventCall.sgpFunctionTable = sgpFunctionTable;
  eventCall.r2 = (void *)0x19;
  eventCall.r3 = eventCall.r4 = eventCall.r6 = 0x0;
  eventCall.r5 = (MInfo *)mInfo;
  MidiEventCall(&eventCall);
  
  WavData wavData;
  wavData.L = L;
  wavData.R = R;  
  ReadData readData;
  readData.sgpFunctionTable = sgpFunctionTable;
  readData.r2 = 0x0;
  readData.r3 = &wavData;
  readData.size = size;
  MidiReadData(&readData);
  delete[] midiMessages;
  delete[] mInfo;
  return 0;
}

void libSGP::dumpLR(unsigned char * L, unsigned char * R, int size, FILE * fp)
{
  for(int i = 0;i < size*4; i+=4)
    {
      fprintf(fp, "%c%c%c%c%c%c%c%c",
              L[i], L[i+1], L[i+2], L[i+3],
              R[i], R[i+1], R[i+2], R[i+3]);
    }
}

void libSGP::dumpLR_S24(unsigned char * L, unsigned char * R, int size, FILE * fp)
{
  // 0xffffff
  long int sL, sR;
  unsigned char d[6];
  float * fL = (float*)L;
  float * fR = (float*)R;
  static float max = 1.0f;
  for(int i = 0;i < size; i++)
    {
      if(fL[i] > max)
	max = fL[i];
      if(fR[i] > max)
	max = fR[i];
      sL = (long int)(fL[i]*(float)(0x800000));
      sR = (long int)(fR[i]*(float)(0x800000));
      memcpy(&(d[0]),&sL,3);
      memcpy(&(d[3]),&sR,3);
      fprintf(fp, "%c%c%c%c%c%c",
              d[0],d[1],d[2],d[3],d[4],d[5]);
    }
}

void libSGP::dumpLR_S32(unsigned char * L, unsigned char * R, int size, FILE * fp)
{
  // 0xffffff
  long int sL, sR;
  unsigned char d[8];
  float * fL = (float*)L;
  float * fR = (float*)R;
  static float max = 1.0f;
  for(int i = 0;i < size; i++)
    {
      if(fL[i] > max)
	max = fL[i];
      if(fR[i] > max)
	max = fR[i];
      sL = (long int)(fL[i]*(float)0x80000000/max);
      sR = (long int)(fR[i]*(float)0x80000000/max);
      memcpy(&(d[0]),&sL,4);
      memcpy(&(d[4]),&sR,4);
      fprintf(fp, "%c%c%c%c%c%c%c%c",
              d[0],d[1],d[2],d[3],d[4],d[5],d[6],d[7]);
    }
}
