// Reverb model implementation
//
// Written by Jezar at Dreampoint, June 2000
// http://www.dreampoint.co.uk
// This code is public domain

#include "revmodel.hpp"

revmodel::revmodel()
{
  isBufferValid = 0;
  allocBuffer(1);
  setBuffer(1);
  setDefault();
  // Buffer will be full of rubbish - so we MUST mute them
  mute();
}

revmodel::~revmodel()
{
  freeBuffer();
}

void revmodel::resetfs(int fs)
{
  float tf = (float)fs/(float)defaultfs;
  int tfs = (int)tf;
  fprintf(stderr, "freeverb: Fs=%d[Hz] Factor=%f(->%d)\n", fs, tf, tfs);
  freeBuffer();
  allocBuffer(tfs);
  setBuffer(tfs);
  mute();
}

void revmodel::printconfig()
{
  fprintf(stderr, "*** Freeverb config ***\n");
  fprintf(stderr, "gain %f roomsize %f roomsize1 %f\n",
	  gain, roomsize,roomsize1);
  fprintf(stderr, "damp %f damp1 %f wet %f wet1 %f wet2 %f\n",
	  damp,damp1,wet,wet1,wet2);
  fprintf(stderr, "dry %f width %f mode %f\n", dry,width,mode);
}

void revmodel::mute()
{
  if (getmode() >= freezemode)
    return;
  
  for (int i=0;i<numcombs;i++)
    {
      combL[i].mute();
      combR[i].mute();
    }
  for (int i=0;i<numallpasses;i++)
    {
      allpassL[i].mute();
      allpassR[i].mute();
    }
}

void revmodel::processreplace(float *inputL, float *inputR, float *outputL, float *outputR,
			      long numsamples, int skip)
{
  float outL,outR,input;
  
  while(numsamples-- > 0)
    {
      outL = outR = 0.0;
      input = (*inputL + *inputR) * gain;
      
      // Accumulate comb filters in parallel
      for(int i=0; i<numcombs; i++)
	{
	  outL += combL[i].process(input);
	  outR += combR[i].process(input);
	}
      
      // Feed through allpasses in series
      for(int i=0; i<numallpasses; i++)
	{
	  outL = allpassL[i].process(outL);
	  outR = allpassR[i].process(outR);
	}
      
      // Calculate output REPLACING anything already there
      *outputL = outL*wet1 + outR*wet2 + *inputL*dry;
      *outputR = outR*wet1 + outL*wet2 + *inputR*dry;
      
      // Increment sample pointers, allowing for interleave (if any)
      inputL += skip;
      inputR += skip;
      outputL += skip;
      outputR += skip;
    }
}

void revmodel::processmix(float *inputL, float *inputR, float *outputL, float *outputR, long numsamples, int skip)
{
  float outL,outR,input;
  
  while(numsamples-- > 0)
    {
      outL = outR = 0.0;
      input = (*inputL + *inputR) * gain;
      
      // Accumulate comb filters in parallel
      for(int i=0; i<numcombs; i++)
	{
	  outL += combL[i].process(input);
	  outR += combR[i].process(input);
	}
      
      // Feed through allpasses in series
      for(int i=0; i<numallpasses; i++)
	{
	  outL = allpassL[i].process(outL);
	  outR = allpassR[i].process(outR);
	}
      
      // Calculate output MIXING with anything already there
      *outputL += outL*wet1 + outR*wet2 + *inputL*dry;
      *outputR += outR*wet1 + outL*wet2 + *inputR*dry;
      
      // Increment sample pointers, allowing for interleave (if any)
      inputL += skip;
      inputR += skip;
      outputL += skip;
      outputR += skip;
    }
}

void revmodel::update()
{
  // Recalculate internal values after parameter change
  
  int i;
  
  wet1 = wet*(width/2 + 0.5f);
  wet2 = wet*((1-width)/2);
  
  if (mode >= freezemode)
    {
      roomsize1 = 1;
      damp1 = 0;
      gain = muted;
    }
  else
    {
      roomsize1 = roomsize;
      damp1 = damp;
      gain = fixedgain;
    }
  
  for(i=0; i<numcombs; i++)
    {
      combL[i].setfeedback(roomsize1);
      combR[i].setfeedback(roomsize1);
    }
  
  for(i=0; i<numcombs; i++)
    {
      combL[i].setdamp(damp1);
      combR[i].setdamp(damp1);
    }
}

// The following get/set functions are not inlined, because
// speed is never an issue when calling them, and also
// because as you develop the reverb model, you may
// wish to take dynamic action when they are called.

void revmodel::setroomsize(float value)
{
  roomsize = (value*scaleroom) + offsetroom;
  update();
}

float revmodel::getroomsize()
{
  return (roomsize-offsetroom)/scaleroom;
}

void revmodel::setdamp(float value)
{
  damp = value*scaledamp;
  update();
}

float revmodel::getdamp()
{
  return damp/scaledamp;
}

void revmodel::setwet(float value)
{
  wet = value*scalewet;
  update();
}

float revmodel::getwet()
{
  return wet/scalewet;
}

void revmodel::setdry(float value)
{
  dry = value*scaledry;
}

float revmodel::getdry()
{
  return dry/scaledry;
}

void revmodel::setwidth(float value)
{
  width = value;
  update();
}

float revmodel::getwidth()
{
  return width;
}

void revmodel::setmode(float value)
{
  mode = value;
  update();
}

float revmodel::getmode()
{
  if (mode >= freezemode)
    return 1;
  else
    return 0;
}

void revmodel::allocBuffer(int factor)
{
#ifdef DEBUG
  fprintf(stderr, "revmodel::allocBuffer(%d)\n", factor);
#endif
  if(isBufferValid == 1)
    {
      freeBuffer();
    }
  bufcombL1 = new float[combtuningL1*factor];
  bufcombR1 = new float[combtuningR1*factor];
  bufcombL2 = new float[combtuningL2*factor];
  bufcombR2 = new float[combtuningR2*factor];
  bufcombL3 = new float[combtuningL3*factor];
  bufcombR3 = new float[combtuningR3*factor];
  bufcombL4 = new float[combtuningL4*factor];
  bufcombR4 = new float[combtuningR4*factor];
  bufcombL5 = new float[combtuningL5*factor];
  bufcombR5 = new float[combtuningR5*factor];
  bufcombL6 = new float[combtuningL6*factor];
  bufcombR6 = new float[combtuningR6*factor];
  bufcombL7 = new float[combtuningL7*factor];
  bufcombR7 = new float[combtuningR7*factor];
  bufcombL8 = new float[combtuningL8*factor];
  bufcombR8 = new float[combtuningR8*factor];
  bufallpassL1 = new float[allpasstuningL1*factor];
  bufallpassR1 = new float[allpasstuningR1*factor];
  bufallpassL2 = new float[allpasstuningL2*factor];
  bufallpassR2 = new float[allpasstuningR2*factor];
  bufallpassL3 = new float[allpasstuningL3*factor];
  bufallpassR3 = new float[allpasstuningR3*factor];
  bufallpassL4 = new float[allpasstuningL4*factor];
  bufallpassR4 = new float[allpasstuningR4*factor];
  isBufferValid = 1;
}

void revmodel::freeBuffer()
{
#ifdef DEBUG
  fprintf(stderr, "revmodel::freeBuffer()\n");
#endif
  if(isBufferValid != 1)
    return;
  delete[] bufcombL1;
  delete[] bufcombR1;
  delete[] bufcombL2;
  delete[] bufcombR2;
  delete[] bufcombL3;
  delete[] bufcombR3;
  delete[] bufcombL4;
  delete[] bufcombR4;
  delete[] bufcombL5;
  delete[] bufcombR5;
  delete[] bufcombL6;
  delete[] bufcombR6;
  delete[] bufcombL7;
  delete[] bufcombR7;
  delete[] bufcombL8;
  delete[] bufcombR8;
  delete[] bufallpassL1;
  delete[] bufallpassR1;
  delete[] bufallpassL2;
  delete[] bufallpassR2;
  delete[] bufallpassL3;
  delete[] bufallpassR3;
  delete[] bufallpassL4;
  delete[] bufallpassR4;
  isBufferValid = 0;
}

void revmodel::setBuffer(int factor)
{
#ifdef DEBUG
  fprintf(stderr, "revmodel::setBuffer(%d)\n", factor);
#endif
  // Tie the components to their buffers
  combL[0].setbuffer(bufcombL1,combtuningL1*factor);
  combR[0].setbuffer(bufcombR1,combtuningR1*factor);
  combL[1].setbuffer(bufcombL2,combtuningL2*factor);
  combR[1].setbuffer(bufcombR2,combtuningR2*factor);
  combL[2].setbuffer(bufcombL3,combtuningL3*factor);
  combR[2].setbuffer(bufcombR3,combtuningR3*factor);
  combL[3].setbuffer(bufcombL4,combtuningL4*factor);
  combR[3].setbuffer(bufcombR4,combtuningR4*factor);
  combL[4].setbuffer(bufcombL5,combtuningL5*factor);
  combR[4].setbuffer(bufcombR5,combtuningR5*factor);
  combL[5].setbuffer(bufcombL6,combtuningL6*factor);
  combR[5].setbuffer(bufcombR6,combtuningR6*factor);
  combL[6].setbuffer(bufcombL7,combtuningL7*factor);
  combR[6].setbuffer(bufcombR7,combtuningR7*factor);
  combL[7].setbuffer(bufcombL8,combtuningL8*factor);
  combR[7].setbuffer(bufcombR8,combtuningR8*factor);
  allpassL[0].setbuffer(bufallpassL1,allpasstuningL1*factor);
  allpassR[0].setbuffer(bufallpassR1,allpasstuningR1*factor);
  allpassL[1].setbuffer(bufallpassL2,allpasstuningL2*factor);
  allpassR[1].setbuffer(bufallpassR2,allpasstuningR2*factor);
  allpassL[2].setbuffer(bufallpassL3,allpasstuningL3*factor);
  allpassR[2].setbuffer(bufallpassR3,allpasstuningR3*factor);
  allpassL[3].setbuffer(bufallpassL4,allpasstuningL4*factor);
  allpassR[3].setbuffer(bufallpassR4,allpasstuningR4*factor);
}

void revmodel::setDefault()
{
  // Set default values
  allpassL[0].setfeedback(0.5f);
  allpassR[0].setfeedback(0.5f);
  allpassL[1].setfeedback(0.5f);
  allpassR[1].setfeedback(0.5f);
  allpassL[2].setfeedback(0.5f);
  allpassR[2].setfeedback(0.5f);
  allpassL[3].setfeedback(0.5f);
  allpassR[3].setfeedback(0.5f);
  setwet(initialwet);
  setroomsize(initialroom);
  setdry(initialdry);
  setdamp(initialdamp);
  setwidth(initialwidth);
  setmode(initialmode);
}

//ends
