/*======================================================================/
  
A driver for the MACNICA SCSI mPS110 card

mPS110 is very similar to qlogic scsi card.
mPS110 may be used a SYM53C500 SCSI protocol controller, which is
similar to NCR53C406 ISA SCSI protocol controller.

This driver referred some parts of following code.
qlogic.c: qlogic linux driver
written by Tom Zerucha,(zerucha@shell.portal.com)

qlogic_cs.c: qlogic pcmcia scsi driver
written by David Hinds,(dhinds@allegro.stanford.edu)

This software may be used and distributed according to the terms of
the GNU Public License.

Written by
N.Katayama (kata-n@po.iijnet.or.jp)
Modified by
Teru KAMOGASHIRA (teru@sodan.ecc.u-tokyo.ac.jp)

======================================================================*/

#include <linux/module.h>
#include <linux/init.h>
#include <linux/stat.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/major.h>
#include <linux/blkdev.h>
#include <linux/delay.h>
#include <linux/types.h>
#include <linux/workqueue.h>

#include <asm/io.h>
#include <asm/system.h>
#include <scsi/scsi.h>
#include <scsi/scsi_ioctl.h>
#include <scsi/scsi_host.h>

#include <../drivers/scsi/scsi.h>

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/ciscode.h>

// #define PC_DEBUG 1

static char *version =
"mps110_cs.c 2.0 [alpha] (N.Katayama, Teru KAMOGASHIRA)";

static int mps110_info(struct Scsi_Host *host, char *buffer, char **start,
		       off_t  offset, int length, int inout);
int mps110_queuecommand(Scsi_Cmnd *, void (* done)(Scsi_Cmnd *));
int mps110_reset(Scsi_Cmnd *);
int mps110_biosparam(struct scsi_device *, struct block_device *, sector_t,int *);
static unsigned int mp_pcmd(Scsi_Cmnd * );
static void mps_interrupt( void * );

static Scsi_Host_Template driver_template;

#ifndef NULL
#define NULL (0)
#endif

/* 53C9x family command */
#define NOP_53C                 0x00
#define CLR_FIFO                0x01
#define SOFT_RESET              0x02
#define SCSI_RESET              0x03
#define CLEAR_ACK               0x12
#define DMA_TRANSFER_INFO       0x90
#define RECEIVE_MSG             0x28
#define SEND_MESSAGE            0x20
#define SET_ATN                 0x1a
#define TRANSFER_INFO           0x10
#define GET_STAT		0x11

/* 53C9x register */
#define TC_LOW          0x00    /* transfer counter LSB         */
#define TC_HIGH         0x01    /* transfer counter MSB         */
#define FIFO            0x02    /* FIFO register                */
#define CMD_REG         0x03    /* command register             */
#define STAT_REG        0x04    /* status register              */
#define DEST_ID         0x04    /* selection/reselection BUS ID */
#define INT_REG         0x05    /* interrupt status register    */
#define SRTIMOUT        0x05    /* select/reselect timeout reg  */
#define SEQ_REG         0x06    /* sequence step register       */
#define SYNCPRD         0x06    /* synchronous transfer period  */
#define FLAGS_REG       0x07    /* indicates # of bytes in fifo */
#define SYNCOFF         0x07    /* synchronous offset register  */
#define CONFIG1         0x08    /* configuration register       */
#define CLKCONV         0x09    /* clock conversion reg         */
#define TESTREG         0x0a    /* test mode register           */
#define CONFIG2         0x0b    /* Configuration 2 Register     */
#define CONFIG3         0x0c    /* Configuration 3 Register     */
#define FIFO_BOTTOM     0x0f    /* Reserve FIFO byte register   */

#define LUN_LIMIT       0x1

/*====================================================================*/
/* Parameters that can be set with 'insmod' */
/* Bit map of interrupts to choose from */
/* This driver never use IRQ */ 
static u_long irq_mask = 0x0000;
/* if you not use MO, set fast_pio = 1 and sg_tablesize = 64 */
static int	fast_pio = 1;
static int	sg_tablesize = SG_ALL;

/*====================================================================*/

typedef struct scsi_info_t {
  int		ndev;
  dev_node_t	node[8];
  struct Scsi_Host *host;
} scsi_info_t;

static void mps110_release(u_long arg);
static int mps110_event(event_t event, int priority, event_callback_args_t *args);
static dev_link_t *mps110_attach(void);
static void mps110_detach(dev_link_t *);

static dev_link_t *dev_list = NULL;

static dev_info_t dev_info = "mps110_cs";

#define PROC_SCSI_NCR53C406A 18000

struct proc_dir_entry proc_scsi_mps110 = {
  PROC_SCSI_NCR53C406A, 6, "mps110",
  S_IFDIR | S_IRUGO | S_IXUGO, 2
};

/*----------------------------------------------------------------*/
/* driver state info, local to driver */
static int	    mbase = 0;	/* Port */
static int	    minitid;	/* initiator ID */
static int	    mabort;	/* Flag to cause an abort */
static int	    mpirq = -1;	/* IRQ being used */
static Scsi_Cmnd   *mpcmd;	/* current command being processed */
// Not a good way to handle interrupt.
static DECLARE_WORK(mps110_work, mps_interrupt, 0); /* task queue entry */

/*----------------------------------------------------------------*/
/* The mps110 card uses two register maps - These macros select which one */
/*----------------------------------------------------------------*/
/* 24bit transfer counter */
#define REG0 ( outb( inb( mbase + 0xd ) & 0x7f , mbase + 0xd ), outb( 4 , mbase + 0xd ))
#define REG1 ( outb( inb( mbase + 0xd ) | 0x80 , mbase + 0xd ), outb( 0xb4, mbase + 0xd ))

/* 16bit transfer counter */
/*
#define REG0 ( outb( 4 , mbase + 0xd ))	
#define REG1 ( outb( 0x80, mbase + 0xd ))
*/

/* following is watchdog timeout in microseconds */
#define WATCHDOG 10000000

/*----------------------------------------------------------------*/
/* local functions */
/*----------------------------------------------------------------*/
static inline void	mp_zap(void);
/* error recovery - reset everything */
void	mp_zap()
{
  int	x;
  unsigned long	flags;
  // TODO
  local_save_flags( flags );
  local_irq_disable();
  
  x = inb(mbase + 0xd);
  REG0;
  outb(SCSI_RESET, mbase + CMD_REG);	/* reset SCSI */
  outb(SOFT_RESET, mbase + CMD_REG);	/* reset chip */
  if (x & 0x80) REG1;
  local_irq_restore( flags );
}

static void mps_interrupt( void *data )
{
  Scsi_Cmnd *cmd;
  int k;
  cmd = mpcmd;
  if( !cmd ) return;
  if( mabort ) {
    if( mabort == 1 ) {
      cmd->result = DID_ABORT << 16;
    } else {
      cmd->result = DID_RESET << 16;
    }
    mpcmd = NULL;
    cmd->scsi_done(cmd);
    return;
  }
  if( !(k=inb(mbase+STAT_REG) & 0xe0) ) {
    // TODO
    schedule_work(&mps110_work);
    
    return;
  }
  if (k & 0x60) mp_zap();
  if (k & 0x20) {
    cmd->result = DID_PARITY << 16;
    mpcmd = NULL;
    cmd->scsi_done(cmd);
    return;
  }
  if (k & 0x40) {
    cmd->result = DID_ERROR << 16;
    mpcmd = NULL;
    cmd->scsi_done(cmd);
    return;
  }
  cmd->result = mp_pcmd(cmd);
  mpcmd = NULL;
  cmd->scsi_done(cmd);
  return;
}

/*----------------------------------------------------------------*/
/* do pseudo-dma                                                  */
/*----------------------------------------------------------------*/
static inline int mp_pdma(int phase, char *request, int reqlen)
{
    int	j;
    j = 0;
    if (phase & 1) {	/* in */
	/* empty fifo in large chunks */
	if( fast_pio ) {
	    if( reqlen >= 128 && (inb( mbase + 8 ) & 2) ) { /* full */
		insl( mbase + 4, request, 32 );
		reqlen -= 128;
		request += 128;
	    }
	    while( reqlen >= 84 && !( j & 0xc0 ) ) /* 2/3 */
		if( (j=inb( mbase + 8 )) & 4 ) {
		    insl( mbase + 4, request, 21 );
		    reqlen -= 84;
		    request += 84;
		}
	    if( reqlen >= 44 && (inb( mbase + 8 ) & 8) ) {	/* 1/3 */
		insl( mbase + 4, request, 11 );
		reqlen -= 44;
		request += 44;
	    }
	}
	/* until both empty and int (or until reclen is 0) */
	j = 0;
	while( reqlen && !( (j & 0x10) && (j & 0xc0) ) ) {
	    /* while bytes to receive and not empty */
	    j &= 0xc0;
	    while ( reqlen && !( (j=inb(mbase + 8)) & 0x10 ) ) {
		*request++ = inb(mbase + 4);
		reqlen--;
	    }
	    if( j & 0x10 )
		    j = inb(mbase+8);

	}
    } else {	/* out */
	if( fast_pio ) {
	    if( reqlen >= 128 && inb( mbase + 8 ) & 0x10 ) { /* empty */
		outsl(mbase + 4, request, 32 );
		reqlen -= 128;
		request += 128;
	    }
	    while( reqlen >= 84 && !( j & 0xc0 ) ) /* 1/3 */
		if( !((j=inb( mbase + 8 )) & 8) ) {
		    outsl( mbase + 4, request, 21 );
		    reqlen -= 84;
		    request += 84;
		}
	    if( reqlen >= 40 && !(inb( mbase + 8 ) & 4 ) ) { /* 2/3 */
		outsl( mbase + 4, request, 10 );
		reqlen -= 40;
		request += 40;
	    }
	}
	/* until full and int (or until reclen is 0) */
	j = 0;
	while( reqlen && !( (j & 2) && (j & 0xc0) ) ) {
	    /* while bytes to send and not full */
	    while ( reqlen && !( (j=inb(mbase + 8)) & 2 ) ) {
		outb(*request++, mbase + 4);
		reqlen--;
	    }
	    if( j & 2 )
		j = inb(mbase+8);
	}
    }
/* maybe return reqlen */
    return inb( mbase + 8 ) & 0xc0;
}

/*----------------------------------------------------------------*/
/* wait for interrupt flag (polled - not real hardware interrupt) */
/*----------------------------------------------------------------*/
static inline int mp_wai(void)
{
  int i,k=0;
  /*
    unsigned long	    flags;
  */
  i = jiffies + WATCHDOG;
  /*
    save_flags( flags );
    sti();
  */
  while ( i > jiffies && !mabort && !((k = inb(mbase + STAT_REG)) & 0xe0)) {
    barrier();
  }
  /*
    restore_flags( flags );
  */
  if (i <= jiffies) {
    return (DID_TIME_OUT);
  }
  if (mabort) 
    return (mabort == 1 ? DID_ABORT : DID_RESET);
  if (k & 0x60)
    mp_zap();
  if (k & 0x20)
    return (DID_PARITY);
  if (k & 0x40)
    return (DID_ERROR);
  return 0;
}

/*----------------------------------------------------------------*/
/* initiate scsi command                                          */
/*----------------------------------------------------------------*/
static inline void mp_icmd(Scsi_Cmnd * cmd)
{
  unsigned int	    i;
  unsigned long	    flags;
  
  
  mabort = 0;
  // TODO
  local_save_flags( flags );
  local_irq_disable();
  
  REG0;
  inb( mbase + INT_REG );
  if( inb( mbase + INT_REG) ) {
    outb(SOFT_RESET, mbase + CMD_REG);	/* reset chip */
    outb(NOP_53C, mbase + CMD_REG);		/* nop */
  } else if( inb( mbase + 7 ) & 0x1f ) {
    outb(CLR_FIFO, mbase + CMD_REG );	/* clear fifo */
  }
  while( inb(mbase + INT_REG) );		/* clear intr. */
  
  /*
    outb( 0x40 | minitid | 0x10 , mbase + CONFIG1);
  */
  outb( minitid, mbase + CONFIG1);
  
  outb( 8, mbase + CLKCONV);		/* clock conv. factor 40MHz/5 */
  /*    outb( 5, mbase + CLKCONV);		/+ clock conv. factor 25MHz/5 */
  outb(0x48, mbase + CONFIG2 );
  outb(0x08, mbase + CONFIG3 );
  
  REG1;
  outb(0x00, mbase + 0x09);		/* Disable IRQ ???? */
  outb(0, mbase + 0x0b );
  outb(1, mbase + 0x08);		/* set for PIO pseudo DMA */
  inb(mbase + 8); 			/* clear int bits */
  REG0;
  outb(0x40, mbase + 0x0b );
  
  // cmd->target ? cmd->device->id
  outb(cmd->device->id , mbase + DEST_ID );	/* 0x04 */
  
  outb(0x99, mbase + SRTIMOUT);	/* select/reselect timer */
  outb(0x00, mbase + SYNCOFF);	/* set sync. offset */
  outb(0x05, mbase + SYNCPRD);	/* set sync. priod */

  outb( CLR_FIFO, mbase + CMD_REG ) ;
  outb( 0x80, mbase + FIFO);		/* Identify message (DiscPriv=NO) */
  for (i = 0; i < cmd->cmd_len; i++) {
    outb(cmd->cmnd[i], mbase + FIFO);
  }
  mpcmd = cmd;
  outb(0x42, mbase + CMD_REG);   	/* select and send command with ATN */
  local_irq_restore( flags );
}

/*----------------------------------------------------------------*/
/* process scsi command - usually after interrupt                 */
/*----------------------------------------------------------------*/
static unsigned int	mp_pcmd(Scsi_Cmnd * cmd)
{
  unsigned int	i, j, k;
  unsigned int	result; 	/* ultimate return result */
  unsigned int	status; 	/* scsi returned status */
  unsigned int	message;	/* scsi returned message */
  unsigned int	phase;		/* recorded scsi phase */
  unsigned int	reqlen; 	/* total length of transfer */
  struct scatterlist	*sglist;	/* scatter-gather list pointer */
  unsigned int	sgcount;	/* sg counter */
  
  j = inb(mbase + SEQ_REG);
  i = inb(mbase + INT_REG);
  if (i == 0x20) {
    return (DID_NO_CONNECT << 16);
  }
  i |= inb(mbase + INT_REG);	/* the 0x10 bit can be set after the 0x08 */
  if (i != 0x18) {
    printk("mps110_cs: Ops:Bad Interrupt status:%02x\n", i);
    mp_zap();
    return (DID_BAD_INTR << 16);
  }
  j &= 7 ; 
  if(j != 3 && j != 4) {
    printk("mps110_cs: Ops:Bad sequence for command %d, int %02X, cmdleft = %d\n", j, i, inb( mbase+7 ) & 0x1f );
    mp_zap();
    return (DID_ERROR << 16);
  }
  result = DID_OK;
  if (inb(mbase + FLAGS_REG) & 0x1f)		/* if some bytes in fifo */
    outb(CLR_FIFO, mbase + CMD_REG);	/* clear fifo */
  reqlen = cmd->request_bufflen;
  if (reqlen && !((phase = inb(mbase + STAT_REG)) & 6)) {	/* data phase */
    outb(reqlen, mbase + TC_LOW );			/* low-mid xfer cnt */
    outb(reqlen >> 8, mbase + TC_HIGH);		/* low-mid xfer cnt */
    outb(reqlen >> 16, mbase + 0x0e);	/* high xfer cnt */
    /*
      REG1;
      if( phase & 1 ) {
      outb(0x00, mbase + 0x0b);
      } else {
      outb(0x01, mbase + 0x0b);
      }
      REG0;
    */
    outb(DMA_TRANSFER_INFO, mbase + CMD_REG);
    /* PIO pseudo DMA to buffer or sglist */
    REG1;
    if (!cmd->use_sg) {
      /* not use Scatter & gather list */
      mp_pdma(phase, cmd->request_buffer, cmd->request_bufflen);
    } else {
      /* use Scatter & gather list */
      sgcount = cmd->use_sg;
      sglist = cmd->request_buffer;
      while (sgcount--) {
	if (mabort) {
	  REG0;
	  return ((mabort == 1 ? DID_ABORT : DID_RESET) << 16);
	}
	if(mp_pdma(phase, page_address(sglist->page) + sglist->offset, sglist->length)) {
	  break;
	}
	sglist++;
      }
    }
    REG0;
    if ((k = mp_wai())) {
      return (k << 16);
    }
    k = inb(mbase + INT_REG);	/* should be 0x10, bus service */
  }
  k = jiffies + WATCHDOG;
  while ( k > jiffies && !mabort && !(inb(mbase + 4) & 6));	/* wait for status phase */
  if ( k <= jiffies ) {
    mp_zap();
    return (DID_TIME_OUT << 16);
  }
  while (inb(mbase + INT_REG)); 		/* clear pending ints */
  if (mabort)
    return ((mabort == 1 ? DID_ABORT : DID_RESET) << 16);
  
  outb(GET_STAT, mbase + CMD_REG);		/* get status and message */
  if ((k = mp_wai()))
    return (k << 16);
  i = inb(mbase + INT_REG);			/* get chip irq stat */
  j = inb(mbase + FLAGS_REG) & 0x1f;		/* and bytes rec'd */
  status = inb(mbase + FIFO);
  message = inb(mbase + FIFO);
  /* should get function complete int if Status and message, else bus serv if only status */
  if (!((i == 8 && j == 2) || (i == 0x10 && j == 1))) {
    printk("mps110_cs: Ops:Error during status phase, int=%02X, %d bytes recd\n", i, j);
    result = DID_ERROR;
  }
  outb(CLEAR_ACK, mbase + CMD_REG);		/* done, disconnect */
  if ((k = mp_wai()))
    return (k << 16);
  
  /* should get bus service interrupt and disconnect interrupt */
  i = inb(mbase + INT_REG);	/* should be bus service */
  while (!mabort && ((i & 0x20) != 0x20)) {
    barrier();
    i |= inb(mbase + INT_REG);
  }
  if (mabort)
    return ((mabort == 1 ? DID_ABORT : DID_RESET) << 16);
  return (result << 16) | (message << 8) | (status & STATUS_MASK);
}


/*----------------------------------------------------------------*/
/* global functions                                               */
/*----------------------------------------------------------------*/
int mps110_queuecommand(Scsi_Cmnd * cmd, void (*done) (Scsi_Cmnd *))
{
#ifdef PC_DEBUG
  printk("mps110_cs: cmd=%02x, cmd_len=%02x, target=%02x, lun=%02x, bufflen=%d\n",
	 cmd->cmnd[0], cmd->cmd_len, cmd->device->id,
	 cmd->device->lun,  cmd->request_bufflen);
#endif
  if(mpcmd) return 0;
  local_irq_enable();
  cmd->scsi_done = done;
  if( cmd->device->id == minitid ) {
    cmd->result = DID_BAD_TARGET << 16;
    mpcmd = NULL;
    done(cmd);
    return 0;
  }
  mp_icmd(cmd);
  // TODO
  schedule_work(&mps110_work);
  
  return 0;
}

/*----------------------------------------------------------------*/
/* return bios parameters */
/*----------------------------------------------------------------*/
int mps110_biosparam(struct scsi_device *sdev,struct block_device *bdev,sector_t cap,int *ip)
{
  struct scsi_device *sd;
  unsigned capacity;
  sd = sdev;
  capacity = cap;
  
  ip[0] = 0x40;
  ip[1] = 0x20;
  ip[2] = capacity / (ip[0] * ip[1]);
  if (ip[2] > 1024) {
    ip[0] = 0xff;
    ip[1] = 0x3f;
    ip[2] = capacity / (ip[0] * ip[1]);
    if (ip[2] > 1023)
      ip[2] = 1023;
  }
  return 0;
}

/*----------------------------------------------------------------*/
/* reset SCSI bus                                                 */
/*----------------------------------------------------------------*/
int mps110_reset(Scsi_Cmnd * cmd)
{
  mabort = 2;
  mp_zap();
  return 1;
}

/*----------------------------------------------------------------*/
/* return info string                                             */
/*----------------------------------------------------------------*/
#undef SPRINTF
#define SPRINTF(args...) \
        do { \
		if(length > (pos - buffer)) { \
			pos += snprintf(pos, length - (pos - buffer) + 1, ## args); \
		} \
	} while(0)

static int mps110_info(struct Scsi_Host *host,
		       char  *buffer,
		       char **start,
		       off_t  offset,
		       int    length,
		       int    inout)
{
  char *pos = buffer;
  int thislength;
  
  SPRINTF("mPS110 status\n\n");
  SPRINTF("SCSI host No.:         %d\n",          host->host_no);
  SPRINTF("IRQ:                   %d\n",          host->irq);
  SPRINTF("IO:                    0x%lx-0x%lx\n", host->io_port, host->io_port + host->n_io_port - 1);
  
  thislength = pos - (buffer + offset);
  if(thislength < 0) {
    *start = NULL;
    return 0;
  }
  thislength = min(thislength, length);
  *start = buffer + offset;
    return thislength;
}
#undef SPRINTF

/*====================================================================*/

static dev_link_t *mps110_attach(void)
{
  client_reg_t client_reg;
  dev_link_t *link;
  int ret;
  
#ifdef PC_DEBUG
  printk(KERN_DEBUG "mps110_cs: mps110_attach\n");
#endif  
  /* Create new SCSI device */
  link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
  memset(link, 0, sizeof(struct dev_link_t));
  link->priv = kmalloc(sizeof(struct scsi_info_t), GFP_KERNEL);
  memset(link->priv, 0, sizeof(struct scsi_info_t));
  // link->release.function = &mps110_release;
  // link->release.data = (u_long)link;
  
  link->io.NumPorts1 = 16;
  link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
  link->io.IOAddrLines = 10;
  link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
  link->irq.IRQInfo1 = IRQ_INFO2_VALID|IRQ_LEVEL_ID;
  link->irq.IRQInfo2 = irq_mask;
  link->conf.Attributes = CONF_ENABLE_IRQ;
  link->conf.Vcc = 50;
  link->conf.IntType = INT_MEMORY_AND_IO;
  link->conf.Present = PRESENT_OPTION;
  
  /* Register with Card Services */
  link->next = dev_list;
  dev_list = link;
  client_reg.dev_info = &dev_info;
  client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
  client_reg.event_handler = &mps110_event;
  client_reg.EventMask =
    CS_EVENT_RESET_REQUEST | CS_EVENT_CARD_RESET |
    CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
    CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
  client_reg.Version = 0x0210;
  client_reg.event_callback_args.client_data = link;
  ret = pcmcia_register_client(&link->handle, &client_reg);
  if (ret != 0) {
    cs_error(link->handle, RegisterClient, ret);
    mps110_detach(link);
    return NULL;
  }
  
  return link;
} /* mps110_attach */

/*====================================================================*/

static void mps110_detach(dev_link_t *link)
{
  dev_link_t **linkp;
  
#ifdef PC_DEBUG
  printk(KERN_DEBUG "mps110_cs: mps110_detach(0x%p)\n", link);
#endif
  /* Locate device structure */
  for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
    if (*linkp == link) break;
  if (*linkp == NULL)
    return;
  
  if (link->state & DEV_CONFIG) {
    mps110_release((u_long)link);
    if (link->state & DEV_STALE_CONFIG) {
      link->state |= DEV_STALE_LINK;
      return;
    }
  }
  
  if (link->handle)
    pcmcia_deregister_client(link->handle);
  
  /* Unlink device structure, free bits */
  *linkp = link->next;
  if (link->priv) {
    kfree(link->priv);
  }
  
  link->priv = NULL;
  kfree(link);
} /* mps110_detach */

/*====================================================================*/
#define CS_CHECK(fn, ret) \
do { last_fn = (fn); if ((last_ret = (ret)) != 0) goto cs_failed; } while (0)

static void mps110_config(dev_link_t *link)
{
  client_handle_t handle = link->handle;
  struct scsi_info_t *info = link->priv;
  tuple_t tuple;
  cisparse_t parse;
  int i, last_ret, last_fn;
  u_char tuple_data[64];
  struct Scsi_Host *host;
  
#ifdef PC_DEBUG
  printk(KERN_DEBUG "mps110_cs: mps110_config(0x%p)\n", link);
#endif  
  
  tuple.TupleData = (cisdata_t *)tuple_data;
  tuple.TupleDataMax = 64;
  tuple.TupleOffset = 0;
  tuple.DesiredTuple = CISTPL_CONFIG;
  CS_CHECK(GetFirstTuple, pcmcia_get_first_tuple(handle, &tuple));
  CS_CHECK(GetTupleData, pcmcia_get_tuple_data(handle, &tuple));
  CS_CHECK(ParseTuple, pcmcia_parse_tuple(handle, &tuple, &parse));
  link->conf.ConfigBase = parse.config.base;
  
  /* Configure card */
  link->state |= DEV_CONFIG;
  
  // usage_count
  // TODO
  //driver_template.present = &mod_use_count_;
  
  tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
  CS_CHECK(GetFirstTuple, pcmcia_get_first_tuple(handle, &tuple));
  while (1) {
    if (pcmcia_get_tuple_data(handle, &tuple) != 0 ||
	pcmcia_parse_tuple(handle, &tuple, &parse) != 0)
      goto next_entry;
    link->conf.ConfigIndex = parse.cftable_entry.index;
    link->io.BasePort1 = parse.cftable_entry.io.win[0].base;
    link->io.NumPorts1 = parse.cftable_entry.io.win[0].len;
    
    if (link->io.BasePort1 != 0) {
      i = pcmcia_request_io(handle, &link->io);
      if (i == CS_SUCCESS)
	break;
    }
  next_entry:
    CS_CHECK(GetNextTuple, pcmcia_get_next_tuple(handle, &tuple));
  }

  CS_CHECK(RequestIRQ, pcmcia_request_irq(handle, &link->irq));
  CS_CHECK(RequestConfiguration, pcmcia_request_configuration(handle, &link->conf));
  
  /* A bad hack... */
  release_region(link->io.BasePort1, link->io.NumPorts1);
  
  printk("mps110_cs: port_base=0x%x+0x%x, irq=%d\n", link->io.BasePort1, link->io.NumPorts1, link->irq.AssignedIRQ);
  
  /* Set port and IRQ */
  mbase = link->io.BasePort1 ;
  mpirq = link->irq.AssignedIRQ;
  
  /* set sg_tablesize */
  if( sg_tablesize >0 && sg_tablesize <= 64 )
    driver_template.sg_tablesize = sg_tablesize;
  
  /* SCSI */
  host = scsi_host_alloc(&driver_template, sizeof(struct scsi_cmnd));
  if (!host) {
    printk("mps110_cs: Unable to register host, giving up.\n");
  }

  host->irq = mpirq;
  host->io_port = mbase;
  host->n_io_port = link->io.NumPorts1;
  // to handle HD, CD-ROM
  host->max_lun = LUN_LIMIT;

  if (scsi_add_host(host, NULL))
    return;
  scsi_scan_host(host);
  
  sprintf(info->node[0].dev_name, "scsi%d", host->host_no);
  link->dev = &info->node[0];
  info->host = host;

  link->state &= ~DEV_CONFIG_PENDING;
  return;

 cs_failed:
  cs_error(link->handle, last_fn, last_ret);
  mps110_release((u_long)link);
  return;
} /* mps110_config */

/*====================================================================*/
static void mps110_release(u_long arg)
{
  dev_link_t *link = (dev_link_t *)arg;
  struct scsi_info_t *info = link->priv;
  struct Scsi_Host *shost = info->host;
  
#ifdef PC_DEBUG
  printk(KERN_DEBUG "mps110_cs: mps110_release(0x%p)\n", link);
#endif
  /*
   *  Do this before releasing/freeing resources.
   */
  scsi_remove_host(shost);

  link->dev = NULL;
  
  pcmcia_release_configuration(link->handle);
  pcmcia_release_io(link->handle, &link->io);
  pcmcia_release_irq(link->handle, &link->irq);
  
  link->state &= ~DEV_CONFIG;
  
  scsi_host_put(shost);
  
  if (link->state & DEV_STALE_LINK)
    mps110_detach(link);
  
} /* mps110_release */

/*====================================================================*/

static int mps110_event(event_t event, int priority,
			event_callback_args_t *args)
{
  dev_link_t *link = args->client_data;
  
#ifdef PC_DEBUG
  printk(KERN_DEBUG "mps110_cs: mps110_event(0x%06x)\n", event);
#endif
  switch (event) {
  case CS_EVENT_REGISTRATION_COMPLETE:
#ifdef PC_DEBUG
    printk(KERN_DEBUG "mps110_cs: registration complete\n");
#endif
    break;
  case CS_EVENT_CARD_REMOVAL:
    link->state &= ~DEV_PRESENT;
    if (link->state & DEV_CONFIG) {
      // link->release.expires = RUN_AT(HZ/20);
      // add_timer(&link->release);
      mps110_release((u_long)link);
    }
    break;
  case CS_EVENT_CARD_INSERTION:
    link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
    mps110_config(link);
    break;
  case CS_EVENT_PM_SUSPEND:
    link->state |= DEV_SUSPEND;
    /* Fall through... */
  case CS_EVENT_RESET_PHYSICAL:
    if (link->state & DEV_CONFIG)
      pcmcia_release_configuration(link->handle);
    break;
  case CS_EVENT_PM_RESUME:
    link->state &= ~DEV_SUSPEND;
    /* Fall through... */
  case CS_EVENT_CARD_RESET:
    if (link->state & DEV_CONFIG) {
      pcmcia_request_configuration(link->handle, &link->conf);
      mps110_reset(NULL);
    }
    break;
  }
  return 0;
} /* mps110_event */

static const char*
mps_info(struct Scsi_Host *SChost)
{
  static char info_msg[256];
  (void)snprintf(info_msg, sizeof(info_msg), "mPS110 at 0x%lx, IRQ %d.", SChost->io_port, SChost->irq);
  return (info_msg);
}

static Scsi_Host_Template driver_template = {
  .proc_name              = "mps110",
  .proc_info              = mps110_info,
  .name                   = "MACNICA MIRACLE SCSI-II mPS110",
  .info                   = mps_info,
  .queuecommand           = mps110_queuecommand,
  .eh_bus_reset_handler   = mps110_reset,
  .bios_param             = mps110_biosparam,
  .slave_configure        = NULL,
  .can_queue              = 1,
  .this_id                = -1,
  .cmd_per_lun            = 1,
  .unchecked_isa_dma      = 0,
  .use_clustering         = DISABLE_CLUSTERING,
  .sg_tablesize           = SG_ALL,
  .present		  = 0,
  .emulated               = FALSE,
};

MODULE_AUTHOR("N.Katayama <kata-n@po.iijnet.or.jp>, Teru KAMOGASHIRA <teru@sodan.ecc.u-tokyo.ac.jp>");
MODULE_DESCRIPTION("MACNICA MIRACLE SCSI-II mPS110 driver");
MODULE_LICENSE("GPL");

static struct pcmcia_driver mps110_cs_driver = {
  .owner          = THIS_MODULE,
  .drv            = {
    .name   = "mps110_cs",
  },
  .attach         = mps110_attach,
  .detach         = mps110_detach,
};

static int __init init_mps110_cs(void)
{
  printk(KERN_INFO "%s\n", version);
  return pcmcia_register_driver(&mps110_cs_driver);
}

static void __exit exit_mps110_cs(void)
{
  pcmcia_unregister_driver(&mps110_cs_driver);
}

module_init(init_mps110_cs);
module_exit(exit_mps110_cs);
